# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''

    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation (optional)
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ---- Exploration Phase: Determine missing predicates, inspect, and identify objects ----
        # The exploration step is necessary because there may be missing knowledge or predicates (e.g., lock status of a drawer or the presence of a dice).
        # We'll assume the robot should systematically check the relevant objects/areas before attempting the manipulation.

        # STEP 1: Inspect/identify all objects and drawers to ensure correct task start
        # For demonstration, let's inspect (visit) all positions to identify objects and drawer state
        for obj_name, obj_pos in positions.items():
            try:
                # Move to each object's position for inspection
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_pos=None,  # Assuming skill can determine from current position if not needed
                    to_pos=obj_pos,
                    max_steps=120,
                    threshold=0.03,
                    timeout=10.0
                )
                # Optionally, identify/inspect the object (not implemented unless a skill exists)
                # Could call execute_go_identify, but skill set only provides execute_* from primitive_skills_static_lock_v2
                # We assume 'execute_go' is sufficient to simulate approaching/inspecting.
            except Exception as e:
                print(f"[Exploration] Failed to inspect {obj_name}: {e}")

        # ---- Task-Specific Manipulation (Assume: Put the single dice into an opened/top drawer) ----
        # The plan: 
        #   1. Ensure the drawer is unlocked 
        #   2. Open the drawer (need to use the handle if required)
        #   3. Ensure hand is free and robot is at dice location, pick the dice
        #   4. Move to the drawer
        #   5. Place the dice inside the drawer
        #   6. Close the drawer

        # Example object keys (you must adapt these to your domain's actual object/drawer names):
        dice_name = None
        dice_pos = None
        drawer_name = None
        drawer_pos = None
        handle_name = None
        handle_pos = None

        # First, try to find the dice, drawer, and handle in positions
        for name in positions:
            if "dice" in name.lower():
                dice_name = name
                dice_pos = positions[name]
            if "drawer" in name.lower() and "handle" not in name.lower():
                drawer_name = name
                drawer_pos = positions[name]
            if "handle" in name.lower():
                handle_name = name
                handle_pos = positions[name]

        # Fallback if any required object is not found
        if dice_name is None or drawer_name is None or handle_name is None:
            print("[Error] Could not find required objects (dice, drawer, handle name missing in positions).")
            return

        # STEP 2: Move to the handle location (preparation for opening the drawer)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=None,  # use current position
                to_pos=handle_pos,
                max_steps=80,
                threshold=0.03,
                timeout=8.0
            )
        except Exception as e:
            print(f"[Task] Failed to go to handle: {e}")
            return

        # STEP 3: Pick the handle (in preparation for pulling/opening)
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=handle_name,
                location=handle_pos,
                approach_distance=0.09,
                max_steps=120,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Failed to pick handle: {e}")
            return

        # STEP 4: Pull to open the drawer (may require: holding handle, drawer must be unlocked and closed)
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                h=handle_name,
                p=handle_pos,
                max_steps=60,
                threshold=0.03,
                timeout=6.0
            )
        except Exception as e:
            print(f"[Task] Failed to pull/open drawer: {e}")
            return

        # STEP 5: Move to the dice location
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=None,  # use current position
                to_pos=dice_pos,
                max_steps=90,
                threshold=0.03,
                timeout=8.0
            )
        except Exception as e:
            print(f"[Task] Failed to move to dice: {e}")
            return

        # STEP 6: Pick the dice (ensure hand is empty, robot is free, robot at dice)
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=dice_name,
                location=dice_pos,
                approach_distance=0.09,
                max_steps=120,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Failed to pick dice: {e}")
            return

        # STEP 7: Move to the drawer location to prepare for placing
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=None,
                to_pos=drawer_pos,
                max_steps=100,
                threshold=0.03,
                timeout=8.0
            )
        except Exception as e:
            print(f"[Task] Failed to move to drawer for place: {e}")
            return

        # STEP 8: Place the dice in the drawer
        try:
            obs, reward, done = execute_place(
                env,
                task,
                o=dice_name,
                d=drawer_name,
                p=drawer_pos,
                max_steps=120,
                threshold=0.02,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Failed to place dice in drawer: {e}")
            return

        # STEP 9: Push to close the drawer (robot must be at drawer, drawer open, robot free)
        try:
            obs, reward, done = execute_push(
                env,
                task,
                d=drawer_name,
                p=drawer_pos,
                max_steps=60,
                threshold=0.03,
                timeout=6.0
            )
        except Exception as e:
            print(f"[Task] Failed to push/close drawer: {e}")
            return

        print("[Task] Successfully put the dice into the drawer and closed it.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
