# run_skeleton_task.py (Filled out with exploration and PDDL-informed logic)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE - Try to discover missing predicates by trying all available skills ===

        # Print object and drawer positions for debugging
        print("[Exploration] Current object positions:", positions)

        # The following is for demonstration and would need to be
        # replaced with your actual environment's objects, drawers, and locations!
        objects = []
        drawers = []
        handles = []
        locations = []
        robot_location = None

        for name, pos in positions.items():
            if "drawer" in name and "handle" not in name:
                drawers.append(name)
            elif "handle" in name:
                handles.append(name)
            elif "loc" in name or "table" in name or "floor" in name:
                locations.append(name)
            else:
                objects.append(name)

        # (Fallback assignment in case object_positions does not provide explicit location names)
        if not locations:
            locations = ["location1", "location2"]

        # Guess robot starting location
        robot_location = None
        for name in locations:
            if "robot" in name or "start" in name:
                robot_location = name
        if robot_location is None and locations:
            robot_location = locations[0]

        print("[Exploration] Identified objects:", objects)
        print("[Exploration] Identified drawers:", drawers)
        print("[Exploration] Identified handles:", handles)
        print("[Exploration] Identified locations:", locations)
        print("[Exploration] Robot initial location:", robot_location)

        # The feedback says: (drawer-unlocked drawer_top)
        # So, check if "drawer_top" is in drawers or in positions keys.
        drawer_to_check = None
        for d in drawers:
            if "top" in d:
                drawer_to_check = d
                break
        if drawer_to_check is None and drawers:
            drawer_to_check = drawers[0]

        # Try to explore/execute actions related to drawer lock-knowledge
        # Since only predefined skills are allowed, attempt using them to probe state.
        # Try to pull the handle of the drawer and look for potential lock-related errors.
        # Assume there is at least one object on the floor or handle to use.
        pull_handle = None
        # Find a handle associated with the drawer
        for h in handles:
            if drawer_to_check and drawer_to_check in h:
                pull_handle = h
                break
        # If not, just select any handle or object
        if pull_handle is None:
            if handles:
                pull_handle = handles[0]
            elif objects:
                pull_handle = objects[0]

        # Find a suitable place/location to approach
        if not robot_location and locations:
            robot_location = locations[0]

        # Ensure all needed entities are available
        if pull_handle and drawer_to_check and robot_location:
            print(f"[Exploration] Trying to pick {pull_handle} at {robot_location}...")
            try:
                # Try picking up the handle (simulate picking handle of drawer)
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=pull_handle,
                    location_name=robot_location
                )
                print("[Exploration] execute_pick succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

            # Try pulling the drawer using the handle
            print(f"[Exploration] Trying to pull drawer {drawer_to_check} with handle {pull_handle}...")
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_to_check,
                    handle_name=pull_handle,
                    location_name=robot_location
                )
                print("[Exploration] execute_pull succeeded.")
                print("[Exploration] Drawer may be unlocked.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")
                print("[Exploration] Possible missing/unmet predicate (e.g., drawer-unlocked).")

        else:
            print("[Exploration] Not enough entities found to attempt skill actions.")

        # --- END EXPLORATION PHASE ---

        # After exploration, your real plan execution happens here!
        # (But per instruction, focus on exploration logic and predicate identification.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
