# run_skeleton_task.py (Completed for Exploration Phase to Find Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use primitives directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Get all current object/handle/drawer/locations
        positions = get_object_positions()

        # Feedback: (holding drawer_top_handle)
        # The system must determine which predicate is missing to allow the robot to reach a (holding drawer_top_handle) state.
        # The 'exploration' domain suggests that we should use execute_pull (and possibly exploration-type actions) to discover if "lock-known" is needed
        # However, we are to use ONLY predefined skills.

        # === 1. Exploration Phase for Missing Predicate ===
        # Simulate attempts to pick/hold the handle. If this fails, use execute_pull (if allowed) to progress the state.
        # We will log the steps for diagnosis and attempt to achieve 'holding drawer_top_handle'.

        # Let's determine handles and drawers from available positions
        handle_names = [name for name in positions if 'handle' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'counter' in name or 'table' in name]

        # For this scenario, try to identify one handle and one drawer
        if not handle_names or not drawer_names:
            print("[Exploration] Could not find handles or drawers in object positions!")
            return
        
        handle_name = handle_names[0]
        handle_pos = positions[handle_name]
        drawer_name = None
        # Attempt to find drawer for handle
        for d in drawer_names:
            if d in handle_name or handle_name in d:
                drawer_name = d
                break
        if not drawer_name:
            drawer_name = drawer_names[0]
        drawer_pos = positions[drawer_name]

        # Try to select a robot starting location as well
        if location_names:
            start_location = location_names[0]
        else:
            # Use any position (could default to handle location)
            start_location = None
        # Fallback: try handle location
        target_location = start_location if start_location else handle_name

        print(f"[Exploration] Attempt to go to location: {target_location}")

        # First, move robot to the handle's location
        # Use execute_go skill: execute_go(from, to)
        try:
            # Depending on the impl, get current robot location (could get from observation or assume 'robot_home' or similar)
            # For the purposes of this, assume start_location is robot's initial location
            # We'll try to go to handle location
            obs, reward, done = execute_go(env, task, from_location=target_location, to_location=handle_name)
            print(f"[Exploration] Robot moved from '{target_location}' to '{handle_name}'")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # Now, try to pick the handle (to result in holding drawer_top_handle)
        try:
            # execute_pick(object, location)
            obs, reward, done = execute_pick(env, task, object_name=handle_name, location_name=handle_name)
            print(f"[Exploration] Attempted pick on handle '{handle_name}' at location '{handle_name}'")
            # Check result - is robot now holding handle?
            # We assume the env/obs indicates this. If so, exploration can end.
        except Exception as e:
            print(f"[Exploration] First execute_pick failed: {e}")
            print(f"[Exploration] Testing for missing predicate via alternate actions.")

        # If pick fails, perhaps the drawer must be unlocked or a precondition is not met.
        # Try to see if any "unlock" or "pull" actions are available in available skills
        # In domain: execute_pull (on closed drawer) unlocks (if requirements met)
        # Try to use execute_pull on the drawer.
        try:
            # execute_pull(drawer, handle, location)
            obs, reward, done = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location_name=handle_name)
            print(f"[Exploration] Tried execute_pull on drawer '{drawer_name}' using handle '{handle_name}' at '{handle_name}'")
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # After pulling, attempt execute_pick again
        try:
            obs, reward, done = execute_pick(env, task, object_name=handle_name, location_name=handle_name)
            print(f"[Exploration] Retried pick on handle '{handle_name}'.")
        except Exception as e:
            print(f"[Exploration] Second execute_pick failed: {e}")
            print(f"[Exploration] Missing predicate likely prevents (holding {handle_name}).")

        print(f"[Exploration] Exploration phase completed. If 'holding {handle_name}' is still false, the missing predicate is likely a lock-related (lock-known ...) or similar precondition.")

        # TODO: Add plan to achieve task goal using the now-learned predicates if possible

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
