# run_skeleton_task.py (Fully Generic Skeleton, now completed for predicate exploration):

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # For exploration, you may need the full positions dict

        # === Exploration Phase for Missing Predicate Identification ===
        # Since the previous planner timed out, we need to explore and find which predicate/condition is required or missing.

        # Strategy:
        # 1. Iterate through relevant locations and objects.
        # 2. Try executing available skill primitives step by step in the environment, observing any failures.
        # 3. Record/log which skill (action) at which step fails—this typically points to a missing predicate in the domain/problem PDDL.

        # ----- Retrieve Environment Info -----
        # Assume 'positions' keys include all objects, drawers, handles, and locations. Let's try to collect them:
        objects = [k for k, v in positions.items() if "obj" in k or "ball" in k or "object" in k]
        drawers = [k for k, v in positions.items() if "drawer" in k and not "handle" in k]
        handles = [k for k, v in positions.items() if "handle" in k]  # handle-of mapping may be needed
        locations = [k for k, v in positions.items() if "location" in k or "room" in k or "table" in k or "pos" in k]

        # Fallback: try all keys if type not strictly given
        if not objects:
            objects = [k for k in positions.keys() if k not in drawers and k not in handles and k not in locations]
        if not locations:
            locations = list(set(positions.keys()) - set(objects) - set(drawers) - set(handles))

        # --- Initial robot location (try to pick from observation if available) ---
        robot_location = None
        for k in locations:
            if "robot" in k or "init" in k:
                robot_location = k
                break
        if not robot_location and locations:
            robot_location = locations[0]  # fallback

        print(f"[Exploration] Objects: {objects}")
        print(f"[Exploration] Drawers: {drawers}")
        print(f"[Exploration] Handles: {handles}")
        print(f"[Exploration] Locations: {locations}")
        print(f"[Exploration] Robot start at: {robot_location}")

        # ----- Begin Predicate Exploration -----
        # We'll try to exercise all skills in the available skill set.
        # For each action, attempt to execute it with all combinations of objects/locations and observe failures.

        skill_names = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Record failed actions, which typically indicate missing predicate or unmet precondition
        failed_actions = []

        # Try moving between locations (execute_go)
        for frm in locations:
            for to in locations:
                if frm == to:
                    continue
                try:
                    print(f"[Try] execute_go: from {frm} to {to}")
                    obs, reward, done = execute_go(env, task, from_location=frm, to_location=to)
                    print(f"[OK] execute_go from {frm} to {to}")
                    break
                except Exception as e:
                    print(f"[Fail] execute_go from {frm} to {to} -- {e}")
                    failed_actions.append(('execute_go', frm, to, str(e)))

        # Try picking up each object (execute_pick)
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Try] execute_pick: object {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj=obj, location=loc)
                    print(f"[OK] execute_pick {obj} at {loc}")
                    break
                except Exception as e:
                    print(f"[Fail] execute_pick {obj} at {loc} -- {e}")
                    failed_actions.append(('execute_pick', obj, loc, str(e)))

        # Try placing each object into each drawer at each location (execute_place)
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Try] execute_place: object {obj} into drawer {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj=obj, drawer=drawer, location=loc)
                        print(f"[OK] execute_place {obj} into drawer {drawer} at {loc}")
                        break
                    except Exception as e:
                        print(f"[Fail] execute_place {obj} into {drawer} at {loc} -- {e}")
                        failed_actions.append(('execute_place', obj, drawer, loc, str(e)))

        # Try pulling each handle (execute_pull)
        for handle in handles:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Try] execute_pull: drawer {drawer}, handle {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=loc)
                        print(f"[OK] execute_pull {drawer}, {handle} at {loc}")
                        break
                    except Exception as e:
                        print(f"[Fail] execute_pull {drawer}, {handle} at {loc} -- {e}")
                        failed_actions.append(('execute_pull', drawer, handle, loc, str(e)))

        # Try pushing each drawer at each location (execute_push)
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Try] execute_push: drawer {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer=drawer, location=loc)
                    print(f"[OK] execute_push {drawer} at {loc}")
                    break
                except Exception as e:
                    print(f"[Fail] execute_push {drawer} at {loc} -- {e}")
                    failed_actions.append(('execute_push', drawer, loc, str(e)))

        # Try sweeping each object (execute_sweep)
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Try] execute_sweep: object {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj=obj, location=loc)
                    print(f"[OK] execute_sweep {obj} at {loc}")
                    break
                except Exception as e:
                    print(f"[Fail] execute_sweep {obj} at {loc} -- {e}")
                    failed_actions.append(('execute_sweep', obj, loc, str(e)))

        # Try gripper action (if applicable)
        try:
            print(f"[Try] execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            print(f"[OK] execute_gripper")
        except Exception as e:
            print(f"[Fail] execute_gripper -- {e}")
            failed_actions.append(('execute_gripper', str(e)))

        # --- Report Exploration Results ---
        print("\n=== Predicate Exploration Results ===")
        if failed_actions:
            for (action, *args, error) in failed_actions:
                print(f"FAILED: {action}({', '.join(str(a) for a in args)}) -- {error}")
        else:
            print("All actions executed without (caught) errors -- review predicate preconditions and effects.")

        print("\n=== End Predicate Exploration ===")

        # (Optional) Use this feedback to repair or augment PDDL domain/problem files with missing predicates!

        # END OF EXPLORATION

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()