# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use skills directly as specified

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running the robotic exploration task to discover missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing simulation progress
        init_video_writers(obs)

        # Wrap task steps for recording if needed
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()  # { 'drawer1': (x, y, z), 'handle1': (x, y, z), ... }

        # === Exploration Phase: Discover the Missing Predicate (lock-known) ===
        # Based on feedback, our task is to figure out (drawer-unlocked drawer1).
        # To do so, we must explore using the available skills and see if the drawer is locked/unlocked.
        # The 'execute_pull' skill can be used to sense the lock status via exploration.

        # For simplicity, we assume the following mapping:
        # - The robot must move to the drawer's location
        # - Then pick up the handle (if separate handle object exists)
        # - Then try to execute_pull, observing the outcome (whether the drawer is locked/unlocked)
        #
        # Since the names/types of objects & locations may change, we will infer reasonable names.

        # ASSUMPTIONS:
        # positions dictionary contains at least the following keys (or similar):
        #   'drawer1'  : the drawer object
        #   'handle1'  : the handle object attached to the drawer
        #   'floor'    : default robot starting position or allowed location
        #   Positions are stored as (x, y, z)

        # Try to extract relevant positions (fallback to using keys as available)
        drawer_name = None
        handle_name = None
        robot_start = None
        for key in positions:
            if 'drawer' in key and drawer_name is None:
                drawer_name = key
            if 'handle' in key and handle_name is None:
                handle_name = key
            if 'floor' in key:
                robot_start = key
            if robot_start is None and 'start' in key:
                robot_start = key

        # Fallbacks if the above fields are not found, just pick something
        if drawer_name is None:
            drawer_name = list(positions.keys())[0]
        if handle_name is None:
            handle_name = drawer_name  # fallback: maybe no separate handle object
        if robot_start is None:
            robot_start = list(positions.keys())[0]

        # Use the first location as the target 'p' for all skills (as required by skill signatures)
        drawer_pos = positions[drawer_name]
        handle_pos = positions[handle_name]
        robot_start_pos = positions[robot_start]

        # === STEP 1: Move to the drawer location if not already there ===
        try:
            print("[Exploration] Moving robot to the drawer location...")
            # We assume only two locations: current and drawer; otherwise more logic is needed
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_start, 
                to_location=drawer_name,
                max_steps=100
            )
            if done:
                print("[Exploration] Task ended at move step.")
                return
        except Exception as e:
            print("[Warning] execute_go failed or robot already at location:", e)

        # === STEP 2: Pick the handle (if it exists as an object) ===
        try:
            print("[Exploration] Attempting to pick handle:", handle_name)
            obs, reward, done = execute_pick(
                env,
                task,
                obj=handle_name,    # handle object
                p=drawer_name,      # the location (drawer area)
                max_steps=100
            )
            if done:
                print("[Exploration] Task ended during handle pick.")
                return
        except Exception as e:
            print("[Warning] execute_pick failed for handle:", e)

        # === STEP 3: Try to pull the drawer (this acts to discover locked state) ===
        try:
            print("[Exploration] Attempting to pull drawer to discover lock status...")
            obs, reward, done = execute_pull(
                env,
                task,
                d=drawer_name,    # drawer object
                h=handle_name,    # handle object (must be held)
                p=drawer_name,    # location (assumed the area of drawer)
                max_steps=100
            )
            if done:
                print("[Exploration] Task ended during drawer pull; check if lock-known predicate is discovered.")
                return
        except Exception as e:
            print("[Warning] execute_pull failed (drawer might be locked):", e)

        # You can inspect the observation, reward, or use custom logic to check if the (drawer-unlocked ...) predicate
        # was discovered here, depending on your system.

        # === END OF EXPLORATION PHASE ===

        print("[Exploration] Finished. If drawer unlock/lock status is unknown, a missing predicate likely exists.")
        print("===== Exploration Complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()