# run_skeleton_task.py (Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # pre-defined primitives: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ======== Exploration Phase for Missing Predicate ========
        print("[Exploration] Begin predicate exploration...")

        # Get a list of all objects and locations in the scene
        all_objects = []
        all_drawers = []
        all_locations = []

        for obj_name in positions.keys():
            if "drawer" in obj_name:
                all_drawers.append(obj_name)
                all_locations.append(obj_name) # drawers can also be considered locations
            elif "handle" in obj_name:
                all_objects.append(obj_name)
            else:
                all_objects.append(obj_name)
        # Also include any workspace or room types as locations
        # (You might want to add such logic depending on your observation keys.)

        # Try to check predicates related to drawers (e.g., locked/unlocked/open/closed)
        # and push/pull to get feedback.

        # For each drawer, try to perform the following:
        for drawer in all_drawers:
            print(f"[Exploration] Exploring drawer {drawer}...")
            try:
                # Find handle
                handle = None
                for h in all_objects:
                    if "handle" in h and drawer in h:  # match handle to drawer by name
                        handle = h
                        break
                if handle is None:
                    # If handle not named as drawer_handle, try any available handle
                    for h in all_objects:
                        if "handle" in h:
                            handle = h
                            break
                # Get robot's current location
                obs = task.get_observation()
                robot_pos = obs.get('robot_pos', None)
                robot_at = None
                if robot_pos is not None:
                    robot_at = robot_pos
                else:
                    # Fallback to some location key
                    for loc in all_locations:
                        if "workspace" in loc or "table" in loc:
                            robot_at = loc
                            break
                    if robot_at is None:
                        robot_at = all_locations[0]
                
                # 1: Try to go to drawer location if not already there
                if robot_at != drawer:
                    try:
                        print(f"[Exploration] Moving robot to {drawer} from {robot_at}")
                        execute_go(env, robot_at, drawer)
                        robot_at = drawer
                    except Exception as e:
                        print(f"[Exploration] execute_go failed: {e}")
                
                # 2: Try to pick the handle first
                if handle is not None:
                    try:
                        print(f"[Exploration] Attempting to pick up handle {handle} at location {drawer}")
                        execute_pick(env, handle, drawer)
                        print(f"[Exploration] Picked up handle {handle}")
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed: {e}")
                
                # 3: Try to pull the drawer open
                try:
                    print(f"[Exploration] Attempting to pull {drawer} using handle {handle} at location {drawer}")
                    execute_pull(env, drawer, handle, drawer)
                    print(f"[Exploration] Pulled {drawer} open (if possible)")
                except Exception as e:
                    print(f"[Exploration] execute_pull failed: {e}")
                
                # 4: Try to push the drawer closed
                try:
                    print(f"[Exploration] Attempting to push {drawer} closed at location {drawer}")
                    execute_push(env, drawer, drawer)
                    print(f"[Exploration] Pushed {drawer} closed (if possible)")
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

            except Exception as e:
                print(f"[Exploration] Unexpected exception: {e}")

        # 5: Try to pick any other objects on the floor at a location
        # (find objects that seem movable and not handles or drawers)
        for obj in all_objects:
            if "handle" in obj or "drawer" in obj:
                continue
            for loc in all_locations:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at location {loc}")
                    execute_pick(env, obj, loc)
                    print(f"[Exploration] Picked {obj} at {loc}")
                    # Optionally: Try to place in a drawer if open
                    for drawer in all_drawers:
                        try:
                            print(f"[Exploration] Attempting to place {obj} into {drawer} at location {loc}")
                            execute_place(env, obj, drawer, loc)
                            print(f"[Exploration] Placed {obj} in {drawer}")
                        except Exception as e:
                            print(f"[Exploration] execute_place failed: {e}")
                    break # after first pick/place attempt
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # You may also want to try sweep/rotate/gripper if available
        for skill_name in ['execute_sweep', 'execute_rotate', 'execute_gripper']:
            func = globals().get(skill_name, None)
            if func is not None:
                for obj in all_objects:
                    for loc in all_locations:
                        try:
                            print(f"[Exploration] Trying {skill_name} on {obj} at {loc}")
                            if skill_name == "execute_gripper":
                                func(env)
                            else:
                                func(env, obj, loc)
                        except Exception as e:
                            print(f"[Exploration] {skill_name} failed: {e}")

        print("[Exploration] Predicate exploration complete. Check feedback/logs for any missing predicate clues.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()