# run_skeleton_task.py (Completed for Feedback-Based Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use pre-defined primitives directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with exploration for state feedback.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # --- EXPLORATION PHASE TO FIND MISSING PREDICATE (drawer-closed drawer1) ---
        # Feedback indicates: must determine if (drawer-closed drawer1)
        # Using only existing skills: 'execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        # 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'

        # Assume object/drawer/location names and match 'drawer1' and relevant handle/object exist
        # Try to open or interact with drawer1, using skills as per PDDL

        # Example extracted values from the environment; adjust as per your true object keys
        drawer_name = 'drawer1'
        # Find a handle for drawer1
        # We'll look through positions/keys to heuristically find a handle attached to drawer1
        handle_name = None
        for obj in positions:
            if 'handle' in obj and drawer_name in obj:
                handle_name = obj
                break
        if not handle_name:
            # fallback: try expected naming
            handle_name = f'{drawer_name}_handle'
            if handle_name not in positions:
                # Give up on explicit handle; log and continue
                print(f"[Exploration] Warning: Could not identify handle for {drawer_name}, please check object naming.")

        # Find the robot's current location (assuming single location predicate)
        robot_location = None
        # Try a few candidate keys for location info
        for key in positions:
            if 'robot' in key and 'location' in key:
                robot_location = positions[key]
                break
        if not robot_location:
            # Fallbacks
            if 'robot_location' in positions:
                robot_location = positions['robot_location']
            else:
                # Default to an explicit known location
                robot_location = list(positions.values())[0]

        # Find the drawer location (useful for go/move)
        drawer_location = None
        try:
            drawer_location = positions[drawer_name]
        except Exception:
            for k in positions:
                if drawer_name in k:
                    drawer_location = positions[k]
                    break
        if drawer_location is None:
            print(f"[Exploration] Warning: Could not find location for {drawer_name}. Using first location in positions.")
            drawer_location = list(positions.values())[0]

        # Attempt to move to the drawer's location
        print(f"[Exploration] Moving robot to drawer location: {drawer_location}")
        try:
            obs, reward, done = execute_go(env, task, robot_location, drawer_location)
        except Exception as e:
            print(f"Error in execute_go: {e}")

        # Now, we attempt to open or "pull" the drawer to check if it is closed/open/locked, 
        # which should let us infer if (drawer-closed drawer1) is true

        # To pull, the robot needs to be holding the handle
        # 1. Pick up the handle (execute_pick). (on-floor ?o), (hand-empty), (robot-free), (robot-at ?p)
        # Assume the handle is on-floor at the drawer's location (adjust appropriately if your system is different)
        print(f"[Exploration] Attempting to pick up handle: {handle_name} at {drawer_location}")
        try:
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
        except Exception as e:
            print(f"Error in execute_pick: {e}")

        # 2. Try to pull the drawer
        # Pre: (holding ?h), (handle-of ?h ?d), (drawer-unlocked ?d), (drawer-closed ?d), (robot-at ?p)
        print(f"[Exploration] Attempting to pull open {drawer_name} with handle {handle_name} at {drawer_location}")
        try:
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            # If this succeeds, we can infer drawer was closed and unlocked
            print(f"[Exploration] Pull success: {drawer_name} was closed and could be opened.")
            exploration_info = "(drawer-closed {})".format(drawer_name)
        except Exception as e:
            # If this fails, might be because drawer wasn't closed, was locked, or
            # some other precondition. Log error.
            print(f"Error in execute_pull: {e}")
            # Exploration predicate missing: likely (drawer-closed {drawer_name}) not satisfied
            exploration_info = None
            # Could attempt to unlock first if an unlock action existed; here just log

        # Record what we've found
        if exploration_info:
            print(f"[Exploration] Exploration indicates predicate present: {exploration_info}")
        else:
            print(f"[Exploration] Could not verify (drawer-closed {drawer_name}), possible missing or unsatisfied preconditions.")

        # Task continues: at this point, insert your oracle plan execution,
        # now able to make decisions using (drawer-closed drawer1) predicate discovered/explored.

        # (Optional) Example: Place an object into the now-open drawer
        # (Assume object_name and more plan steps as necessary)
        # object_name = 'object1'
        # try:
        #     obs, reward, done = execute_pick(env, task, object_name, drawer_location)
        # except Exception as e:
        #     print(f"Error picking {object_name}: {e}")
        # try:
        #     obs, reward, done = execute_place(env, task, object_name, drawer_name, drawer_location)
        # except Exception as e:
        #     print(f"Error placing {object_name} in {drawer_name}: {e}")

        # Continue with oracle plan logic...

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()