# run_skeleton_task.py (Completed for Exploration & Plan Execution)
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine primitives, must use as is

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Dictionary with positions or other info
        positions = get_object_positions()
        # As objects/drawers/handles must be defined in 'positions':
        # We expect entries like: 'drawer1', 'handle_drawer1', etc.

        # ---------- Exploration Phase: Identify Missing Predicate ----------
        # Feedback tells us (drawer-unlocked drawer1) was missing.
        # The domain requires knowing 'drawer-unlocked' before 'pull'
        #
        # In a realistic simulation, the robot would attempt to check the lock state of drawer1.
        # We use skills to probe this, e.g., by trying to open the drawer (pull).
        # 
        # The 'execute_pull' action in domain PDDL has the precondition 'drawer-unlocked'.
        # If this is not known, we need to perform exploration to discover it.
        #
        # Assuming the exploration skill here would be a sweep, or a dedicated probe action, 
        # but per available_skills, only these are available:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # We use the following logic:
        # 1. Attempt to use 'execute_pull' with the assumption the predicate may be missing
        # 2. If exception is raised due to missing preconditions (i.e., unknown 'drawer-unlocked'), 
        #    perform a sweep or alternative exploration skill if available.
        # 3. After exploration, try 'execute_pull' again to open the drawer.

        # Required objects (update these keys if your positions dict uses different names)
        drawer_name = "drawer1"
        handle_name = "handle_drawer1"  # The handle object for drawer1
        floor_object_name = "object1"   # The target object on the floor
        place_location_name = "place_location1"  # General target for execute_go

        # Defensive loading of key object positions
        try:
            drawer_pos = positions[drawer_name]
            handle_pos = positions[handle_name]
            object_pos = positions[floor_object_name]
            place_location_pos = positions[place_location_name]
        except KeyError as ke:
            print(f"[Error] Key not found in object_positions: {ke}")
            return

        print(f"[Info] Located drawer at: {drawer_pos}, handle at: {handle_pos}, floor object at: {object_pos}")

        # Let's get names consistent for skill_code calls
        # Skill signatures (in skill_code) will likely look like:
        #   execute_pick(env, task, object_name, location)
        #   execute_place(env, task, object_name, drawer_name, location)
        #   execute_pull(env, task, drawer_name, handle_name, location)
        #   execute_push(env, task, drawer_name, location)
        #   execute_go(env, task, from_location, to_location)
        #   execute_sweep(env, task, object_name, location)
        #   execute_gripper(...)
        # If they require more/less args, handle exceptions below.

        current_location = None
        try:
            # Step 1: Move to object and pick up the handle to explore the lock state

            # Move robot to handle location
            print("[Step] Moving robot to handle location...")
            obs, reward, done = execute_go(env, task, "start", handle_name)
            current_location = handle_name
            if done:
                print("[Task] Task ended early during go.")
                return

            # Try to pick the handle (so we can later try to pull/open the drawer)
            print("[Step] Picking handle...")
            obs, reward, done = execute_pick(env, task, handle_name, current_location)
            if done:
                print("[Task] Task ended after picking handle.")
                return

            # Now, try to open the drawer (may fail if drawer-unlocked is missing in init).
            print("[Step] Trying to pull (open) the drawer...")
            try:
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, current_location)
                if done:
                    print("[Task] Task ended after opening drawer.")
                    return
                print("[Info] Drawer successfully opened: predicate 'drawer-unlocked' was satisfied.")
            except Exception as e:
                print(f"[Exploration] Pull failed or precondition not met: {e}")
                print("[Exploration] Attempting to discover lock state of drawer with exploration skill...")

                # If 'execute_sweep' can be used for discovery, let's do so.
                try:
                    obs, reward, done = execute_sweep(env, task, drawer_name, current_location)
                    print("[Exploration] Performed exploration (sweep) to discover lock state.")
                except Exception as e2:
                    print(f"[Warning] Exploration skill failed: {e2}")
                    # If sweep is not appropriate, skip to next best attempt

                # Try pull again, now that we have explored
                try:
                    obs, reward, done = execute_pull(env, task, drawer_name, handle_name, current_location)
                    if done:
                        print("[Task] Task ended after opening drawer following exploration.")
                        return
                    print("[Info] Drawer opened after exploration.")
                except Exception as e3:
                    print(f"[Error] Still cannot open the drawer: {e3}")
                    return

            # Step 2: Move to floor object and pick it
            print("[Step] Moving robot to object on floor...")
            obs, reward, done = execute_go(env, task, current_location, floor_object_name)
            current_location = floor_object_name
            if done:
                print("[Task] Task ended during go to object.")
                return

            # Pick the object
            print("[Step] Picking up floor object...")
            obs, reward, done = execute_pick(env, task, floor_object_name, current_location)
            if done:
                print("[Task] Task ended after picking up floor object.")
                return

            # Step 3: Move to drawer to place the object inside
            print("[Step] Moving to drawer to place object...")
            obs, reward, done = execute_go(env, task, current_location, drawer_name)
            current_location = drawer_name
            if done:
                print("[Task] Task ended during go to drawer.")
                return

            # Place the object in the drawer
            print("[Step] Placing object into drawer...")
            obs, reward, done = execute_place(env, task, floor_object_name, drawer_name, current_location)
            if done:
                print("[Task] Object placed and task ends.")
                return

            # Optionally, close the drawer
            print("[Step] Pushing drawer to close it...")
            obs, reward, done = execute_push(env, task, drawer_name, current_location)
            if done:
                print("[Task] Drawer closed and task ends.")
                return

        except Exception as e:
            print(f"[Fatal Error] Exception during skeleton task: {e}")
            return

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()