# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicate'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # You may need object, drawer, handle, and location names as per your environment
        # This section assumes you have some reference naming convention or can get them from positions
        try:
            drawer_name = 'drawer_main'  # provided from feedback
            # Find handle object associated with this drawer
            # If there is a mapping, otherwise try to infer from positions
            handle_name = None
            for obj, pos in positions.items():
                if 'handle' in obj and 'drawer' in obj:
                    handle_name = obj
                    break
            if not handle_name:
                # Fallback: some other way to infer handle, or try a default
                handle_name = 'drawer_main_handle'
        except Exception as e:
            print(f"[Exploration] Could not resolve handle/drawer: {e}")
            return

        # Assume location positions or names (need at least one for robot location)
        location_names = [k for k in positions.keys() if 'loc' in k or 'floor' in k or 'counter' in k or 'surface' in k]
        if not location_names:
            # fallback default if environment uses classic location names
            location_names = ['floor_left', 'floor_right', 'counter_top', 'drawer_area']
        robot_location = None
        # try to determine robot_start location
        if hasattr(task, 'robot_init_location'):
            robot_location = task.robot_init_location
        else:
            robot_location = location_names[0] if location_names else 'floor_left'
        # Possibly destination for plan/placement, etc.
        destination_location = location_names[1] if len(location_names) > 1 else robot_location

        # === === === MISSING PREDICATE EXPLORATION === === ===

        # Feedback says: (drawer-closed drawer_main) is missing, i.e. we need to find out if the drawer is closed or not
        # This corresponds to the predicate (drawer-closed drawer_main)
        # In practice, exploration would involve interacting with the drawer to determine its closed state.

        # For the purposes of using only provided skills, let's try to:
        # 1. Move to the drawer area if not already there
        # 2. Optionally try to interact (pull/push) with the drawer and observe
        # 3. Call the corresponding exploration action (if available; otherwise, call corresponding skill and check outcome)

        print("[Exploration] Determining if drawer is closed (missing predicate) for:", drawer_name)

        # We assume execute_pull and execute_push are skills, and that we can call them to try the drawer
        # Since the exploration PDDL example includes actions like 'execute_pull' to check lock/closed state
        # But since we cannot call special exploration skills (not in provided skills), we mimic the exploration behavior

        # 1. Move to the drawer location if not already there (requires execute_go skill)
        # (Assume robot is always free at start and robot_location is known)
        try:
            if robot_location != destination_location:
                print(f"[Skill] Moving robot from {robot_location} to {destination_location}")
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=destination_location)
                robot_location = destination_location
                if done:
                    print("[Task] Task ended unexpectedly after move!")
                    return
        except Exception as e:
            print(f"[Skill Error] execute_go: {e}")

        # 2. Try to pick the handle (need to have a free hand)
        try:
            print(f"[Skill] Picking drawer handle: {handle_name} at {robot_location}")
            obs, reward, done = execute_pick(env, task, obj=handle_name, p=robot_location)
            if done:
                print("[Task] Task ended unexpectedly after pick!")
                return
        except Exception as e:
            print(f"[Skill Error] execute_pick: {e}")

        # 3. Try to pull the drawer open (will only succeed if closed and unlocked per PDDL)
        try:
            print(f"[Skill] Attempting to pull/open {drawer_name} using handle {handle_name} at {robot_location}")
            obs, reward, done = execute_pull(env, task, d=drawer_name, h=handle_name, p=robot_location)
            # If this succeeds, the drawer was closed, and now is open. Record this knowledge
            print("[Exploration Result] Drawer pull attempted - outcome indicates if drawer was closed.")
            if done:
                print("[Task] Task ended unexpectedly after pull!")
                return
        except Exception as e:
            print(f"[Skill Error] execute_pull: {e}")

        # Now, either the pull succeeded (drawer is now open), or not (e.g., it was locked or already open)
        # In a more complete scenario, you may try push, or handle error, or record observations, etc.

        # === === === END EXPLORATION === === ===

        print("[Exploration] Completed. Proceeding to oracle plan (if any)...")

        # === Plan execution would proceed here (skipped for this exploration-only problem) ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
