# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills, do NOT redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including an exploration phase to identify missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- EXPLORATION PHASE TO DETECT MISSING PREDICATE ---
        #
        # Based on exploration knowledge, we will attempt to execute available skills 
        # to probe for unmodeled states or missing predicates.
        #
        # We use only predefined skills here.
        #
        # For demonstration, we try combinations of available skills and check for abnormal failures.
        # This helps to collect feedback indicative of missing predicates.

        exploration_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]
        
        # Attempt every skill on each object/location as exploration
        # (Here, we focus on the skills most relevant to lock or unknown conditions)

        print("[Exploration] Starting exploration phase to probe potential missing predicate (e.g., lock state).")
        for obj_name, obj_info in positions.items():
            try:
                # Try pulling this object at its location
                loc = obj_info.get('location', None)
                handle = obj_info.get('handle', obj_name)  # If 'handle' exists, else use object
                drawer = obj_info.get('drawer', None)

                # Try 'execute_pull' if applicable (usually for handles/drawers)
                if 'execute_pull' in exploration_skills and drawer and loc:
                    print(f"[Exploration] Trying execute_pull on {drawer} using handle {handle} at {loc}")
                    try:
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            drawer,
                            handle,
                            loc
                        )
                        if done:
                            print(f"[Exploration] execute_pull succeeded ({drawer}, {handle}, {loc})")
                        else:
                            print(f"[Exploration] execute_pull did NOT end episode")
                    except Exception as exc:
                        print(f"[Exploration] execute_pull failed on ({drawer}, {handle}, {loc}): {exc}")

                # Try picking up the object if it is on floor
                if 'execute_pick' in exploration_skills and loc and obj_info.get('on_floor', False):
                    print(f"[Exploration] Trying execute_pick on {obj_name} at {loc}")
                    try:
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            obj_name,
                            loc
                        )
                        if done:
                            print(f"[Exploration] execute_pick succeeded ({obj_name}, {loc})")
                        else:
                            print(f"[Exploration] execute_pick did NOT end episode")
                    except Exception as exc:
                        print(f"[Exploration] execute_pick failed on ({obj_name}, {loc}): {exc}")
                
                # Try pushing drawers if relevant
                if 'execute_push' in exploration_skills and drawer and loc:
                    print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                    try:
                        obs, reward, done = execute_push(
                            env,
                            task,
                            drawer,
                            loc
                        )
                        if done:
                            print(f"[Exploration] execute_push succeeded ({drawer}, {loc})")
                        else:
                            print(f"[Exploration] execute_push did NOT end episode")
                    except Exception as exc:
                        print(f"[Exploration] execute_push failed on ({drawer}, {loc}): {exc}")

                # Try go action (move robot)
                if 'execute_go' in exploration_skills and loc:
                    # Here, let's try moving from current robot location to this object's location
                    robot_pos = None
                    for name, info in positions.items():
                        if info.get('is_robot', False):
                            robot_pos = info.get('location')
                            break
                    if robot_pos and robot_pos != loc:
                        print(f"[Exploration] Trying execute_go from {robot_pos} to {loc}")
                        try:
                            obs, reward, done = execute_go(
                                env,
                                task,
                                robot_pos,
                                loc
                            )
                            if done:
                                print(f"[Exploration] execute_go succeeded ({robot_pos} -> {loc})")
                            else:
                                print(f"[Exploration] execute_go did NOT end episode")
                        except Exception as exc:
                            print(f"[Exploration] execute_go failed ({robot_pos} -> {loc}): {exc}")

            except Exception as all_exc:
                print(f"[Exploration] Error with object {obj_name}: {all_exc}")

        print("[Exploration] Completed. Analyze output above for missing predicate related errors (e.g., lock).")

        # --- PLAN EXECUTION PHASE (EXAMPLE PLACEHOLDER) ---
        #
        # Here is where you would normally execute the oracle plan step-by-step using predefined skills,
        # using object/location names and arguments parsed from your plan.
        #
        # For example (uncomment and modify as needed per your plan):
        #
        # obs, reward, done = execute_go(env, task, "loc1", "loc2")
        # obs, reward, done = execute_pick(env, task, "objectA", "loc2")
        # obs, reward, done = execute_pull(env, task, "drawerX", "handleY", "loc2")
        # obs, reward, done = execute_place(env, task, "objectA", "drawerX", "loc2")
        #
        # if done:
        #     print("[Task] Task ended after executing the plan.")
        #     return

        # TODO: To use this template for an actual plan, fill in concrete actions below.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
