# run_skeleton_task.py (Completed for Exploration Phase: Missing Predicate Identification)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # (Assume skills: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation with an exploration phase for missing predicate detection.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ---------------- EXPLORATION PHASE ----------------
        # The feedback mentions (robot-at drawer_top_place_left)
        # We try to use available skills to move the robot to that location and log which predicate prevents further planning.
        
        # Assumptions:
        # - "drawer_top_place_left" is one of the locations in the environment.
        # - The robot starts at some known location (let's try to get it from initial positions or descriptions)
        # - The available skill for moving the robot is "execute_go".
        # - We need to check whether moving to "drawer_top_place_left" works and, if not, output/log which predicate is missing.

        # Step 1: Determine the robot's current location and the target location
        robot_start_location = None
        target_location = "drawer_top_place_left"

        # Attempt to find the robot's current location from the "positions" mapping
        # This mapping may look like: {'robot': (x, y, z, 'some_location'), ...}
        # Alternatively, try to get initial location from task/descriptions if not present in positions

        try:
            if 'robot' in positions and isinstance(positions['robot'], (tuple, list)) and len(positions['robot']) >= 4:
                robot_start_location = positions['robot'][3]  # ['robot'] = (x, y, z, location_name)
            elif hasattr(task, 'robot_location'):
                robot_start_location = task.robot_location
            else:
                # Fallback: try to parse descriptions/obs (but usually a string or dict)
                robot_start_location = 'unknown_location'
        except Exception as e:
            print("[Exploration] Could not determine robot's current location:", str(e))
            robot_start_location = 'unknown_location'

        print(f"[Exploration] Robot start location: {robot_start_location}")
        print(f"[Exploration] Target location: {target_location}")

        # Step 2: Try to execute the 'execute_go' skill to move the robot to 'drawer_top_place_left'

        # Defensive: ensure both location strings are set and distinct
        if (robot_start_location is not None
            and robot_start_location != target_location
            and target_location is not None):
            try:
                print(f"[Exploration] Attempting to move from {robot_start_location} to {target_location} using 'execute_go'...")
                # execute_go expects: env, from_location, to_location
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_start_location,
                    to_location=target_location
                )
                print("[Exploration] Move executed. Checking observation for missing predicate...")
                # Optionally, check for a marker in obs indicating predicate failure (example: obs['missing_predicate'])
                missing_predicate = obs.get('missing_predicate', None) if isinstance(obs, dict) else None
                if missing_predicate:
                    print(f"[Exploration] Missing predicate detected: {missing_predicate}")
                else:
                    print("[Exploration] No missing predicate reported in observation. Move might have succeeded.")
                if done:
                    print("[Exploration] Task signaled 'done' after execute_go.")
            except Exception as e:
                # Most likely a predicate failure or precondition not satisfied
                print(f"[Exploration] Exception during move: {str(e)}")
                # Optionally try to infer missing predicate from exception message
                error_msg = str(e).lower()
                if "robot-at" in error_msg:
                    print("[Exploration] Predicate 'robot-at' seems missing or unsatisfied (confirm domain and initial state).")
                elif "location" in error_msg:
                    print("[Exploration] One or both location values are not known in the environment.")
                else:
                    print("[Exploration] An unknown predicate or precondition may be missing. See error above.")

        else:
            print("[Exploration] Unable to attempt move: could not resolve both current and target location.")

        # ---------------- END OF EXPLORATION PHASE ----------------

        # ------------------------------------------
        # NOTE:
        # Any further steps (plan execution based on predicates found, or more manipulation) should be inserted here.
        # For this task, we stop at the exploration phase per requirements.
        # ------------------------------------------

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
