# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only these skill functions; do not redefine
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ======================================================
        #           Exploration: Predicate Discovery
        # ======================================================

        print("=== Exploration Phase: Predicate Discovery ===")
        # Feedback: (robot-at drawer-area)
        # That is, the robot needs to get to 'drawer-area', but 'robot-at drawer-area' predicate is missing or not true.
        # We must attempt to move there and confirm the result using the predefined skills.
        # The relevant skill for movement from the primitive_skills_static_lock_v2 domain is 'execute_go'.
        # We use only imported skills and not define any new skill.
        # We will print all available locations to verify which one matches 'drawer-area'.

        # Let's try to parse the available locations from 'positions' dictionary keys, look for 'drawer' or 'drawer-area'.
        location_candidates = []
        for name in positions.keys():
            if 'drawer' in name or 'area' in name:
                location_candidates.append(name)
        if 'drawer-area' not in location_candidates:
            location_candidates.append('drawer-area')  # Try the canonical name as feedback suggests

        robot_current_location = None
        for name in positions.keys():
            if 'robot' in name or 'base' in name:
                robot_current_location = name
                break
        if robot_current_location is None:
            robot_current_location = 'home'  # fallback to possible default

        print("[Exploration] Robot current location guess:", robot_current_location)
        print("[Exploration] Location candidates for drawer area:", location_candidates)

        # Try to move the robot to drawer-area using execute_go for all plausible variants
        moved = False
        for loc in location_candidates:
            try:
                print(f"[Exploration] Trying to move robot from {robot_current_location} to {loc} using execute_go")
                obs, reward, done = execute_go(env, task, from_location=robot_current_location, to_location=loc)
                if done or reward is not None:  # If the skill indicates successful step
                    print(f"[Exploration] Success moving to {loc}, possible matching predicate: (robot-at {loc})")
                    moved = True
                    break  # Stop at first successful move
            except Exception as e:
                print(f"[Exploration] Attempt to move to {loc} failed: {e}")
        if not moved:
            print("[Exploration] Could not move to any candidate location, missing predicate likely 'robot-at drawer-area' or an alias.")

        # At this point, the missing predicate for planning is likely (robot-at drawer-area).
        # The exploration phase ends here after attempting to confirm which predicate is needed.

        # ======================================================
        #    Task Plan Execution (after Required Predicates)
        # ======================================================
        # This is where you proceed with your normal oracle plan,
        # After the exploration phase you would use the predicates you found to drive the controller logic.
        # Below is an example stub for such a plan.

        # ---- Example Plan Steps (Needs to be replaced by actual Oracle Plan) ----
        # For illustration, we assume a plan:
        #  1. move to drawer-area
        #  2. open the drawer (pull handle)
        #  3. pick object from floor
        #  4. place in drawer
        #  5. push drawer closed

        # For each, use the provided skills, using object/location/handle names from positions dict.
        # You must adapt object names below as per your env & plan.

        # 1. Move to drawer-area (already attempted above in exploration)

        # 2. Pull open the drawer using its handle
        drawer = None
        handle = None
        for name in positions.keys():
            if 'drawer' in name:
                drawer = name
            if 'handle' in name:
                handle = name
        if not drawer:
            drawer = 'drawer'
        if not handle:
            handle = 'handle'

        # 3. Pick an object from the floor
        obj_to_pick = None
        for name in positions.keys():
            if 'obj' in name or 'item' in name or 'cube' in name or 'ball' in name:
                obj_to_pick = name
        if obj_to_pick is None:
            # fallback to first non-drawer, non-handle, non-robot entry
            for name in positions.keys():
                if not any(keyword in name for keyword in ['drawer', 'handle', 'robot', 'base', 'area']):
                    obj_to_pick = name
                    break
        if obj_to_pick is None:
            obj_to_pick = 'object_1'

        # 4. Place in drawer
        # 5. Push drawer closed

        # These steps can only be performed if predicate exploration succeeded
        if moved:
            # 2. Pull open drawer using its handle
            try:
                print(f"[Task] Attempting to pick the handle ({handle}) for pulling drawer ({drawer})...")
                # First: Pick the handle (requires hand empty and robot at correct location)
                obs, reward, done = execute_pick(env, task, obj=handle, location=loc)
                print(f"[Task] Picked handle {handle}.")
            except Exception as e:
                print(f"[Task] Error when picking handle: {e}")
            try:
                print(f"[Task] Pulling the drawer '{drawer}' using handle '{handle}'...")
                # Then: Pull the drawer open
                obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=loc)
                print(f"[Task] Drawer {drawer} pulled open.")
            except Exception as e:
                print(f"[Task] Error when pulling drawer: {e}")

            # 3. Pick up the object from the floor
            try:
                print(f"[Task] Picking object {obj_to_pick} from location {loc}...")
                obs, reward, done = execute_pick(env, task, obj=obj_to_pick, location=loc)
                print(f"[Task] Picked object {obj_to_pick}.")
            except Exception as e:
                print(f"[Task] Error when picking object: {e}")

            # 4. Place the object into the drawer
            try:
                print(f"[Task] Placing object {obj_to_pick} into drawer {drawer} at location {loc}...")
                obs, reward, done = execute_place(env, task, obj=obj_to_pick, drawer=drawer, location=loc)
                print(f"[Task] Placed object in drawer.")
            except Exception as e:
                print(f"[Task] Error when placing object: {e}")

            # 5. Close the drawer
            try:
                print(f"[Task] Pushing (closing) drawer {drawer} at location {loc}...")
                obs, reward, done = execute_push(env, task, drawer=drawer, location=loc)
                print(f"[Task] Drawer closed.")
            except Exception as e:
                print(f"[Task] Error when closing drawer: {e}")
        else:
            print("[Task] Aborting task plan steps due to failure to move to the drawer-area.")

        # === End of Main Task Logic ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
