# run_skeleton_task.py (Completed with Exploration Phase for Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use available skills only; do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ----------------------------
        # Exploration Phase: Find Missing Predicate
        # (e.g., (robot-at drawer-area) as indicated by feedback)
        # ----------------------------

        # Attempt to move the robot to all known locations to see if (robot-at ...) is recognized
        # We expect that 'drawer-area' may be missing as a predicate; explore accordingly.

        known_locations = []
        for obj_name, obj_info in positions.items():
            # Suppose obj_info may be a dict with type info
            if 'location' in obj_info:
                loc = obj_info['location']
                if loc not in known_locations:
                    known_locations.append(loc)
            elif 'type' in obj_info:
                # For location objects
                if obj_info['type'] == 'location' and obj_name not in known_locations:
                    known_locations.append(obj_name)
            else:
                continue

        # Fallback if no locations specified by get_object_positions
        if not known_locations:
            # Hardcode common locations based on PDDL
            known_locations = ['drawer-area']

        missing_predicates = []

        # We'll instantiate the exploration: try to move to every location
        current_loc = None
        # First, try to locate current robot position
        for loc in known_locations:
            try:
                # Does the env/task or obs provide robot location? If so, set as start
                if obs.get('robot_at', None) == loc:
                    current_loc = loc
                    break
            except Exception:
                pass

        # If not found, default to the first location
        if current_loc is None:
            current_loc = known_locations[0]

        print("[Exploration] Known locations for exploration:", known_locations)
        print("[Exploration] Starting from location:", current_loc)

        # Try to move to drawer-area using available skill
        for target_loc in known_locations:
            if target_loc == current_loc:
                continue
            print(f"[Exploration] Attempting to move from {current_loc} to {target_loc}")
            try:
                # Use the available skill: execute_go (move from -> to)
                obs, reward, done = execute_go(env, task, current_loc, target_loc)
                print(f"[Exploration] After move, checking for predicate (robot-at {target_loc})")
                # Check observation for predicate (robot-at target_loc)
                if obs.get('robot_at', None) == target_loc or \
                   obs.get(f'robot_at_{target_loc}', False) or \
                   ("robot-at " + target_loc in str(obs)):
                    print(f"[Exploration] Predicate (robot-at {target_loc}) recognized in observation.")
                else:
                    # If not found, log as missing predicate
                    print(f"[Exploration] Predicate (robot-at {target_loc}) likely missing!")
                    missing_predicates.append(f"(robot-at {target_loc})")
                # Update current_loc
                current_loc = target_loc
            except Exception as e:
                print(f"[Exploration] Exception or failure moving to {target_loc}: {str(e)}")
                missing_predicates.append(f"(robot-at {target_loc})")

        if missing_predicates:
            print("[Exploration] Missing predicates detected:", missing_predicates)
        else:
            print("[Exploration] All predicates detected.")

        # ----------------------------
        # Main Plan Execution (Example, replace with real oracle-plan)
        # Here, you would use only the available skill names to run the main plan.
        # E.g., execute_pick, execute_place, execute_push, execute_pull, etc.
        # (You would extract object, drawer, location names from positions or description)
        # Only call the provided skills directly and in the right order as dictated by the plan.
        # You may wish to check for completion at each step.

        # Example: Suppose oracle plan is:
        # 1. Navigate to 'drawer-area'
        # 2. Open 'drawer1' (if needed), using pull
        # 3. Pick object 'item1'
        # 4. Place in drawer
        # This code stub is only illustrative. Replace argument names with actual ones from your scenario.

        # Assume that after exploration, we know how to move to 'drawer-area'

        try:
            # 1) Move robot to 'drawer-area'
            if current_loc != 'drawer-area':
                print("[Plan] Moving robot to drawer-area...")
                obs, reward, done = execute_go(env, task, current_loc, 'drawer-area')
                current_loc = 'drawer-area'
        except Exception as e:
            print(f"[Plan] Error during move: {str(e)}")

        # 2) Identify handle and drawer names from positions, if possible
        handle_name = None
        drawer_name = None
        for obj_name, obj_info in positions.items():
            if 'type' in obj_info:
                if obj_info['type'] == 'handle':
                    handle_name = obj_name
                if obj_info['type'] == 'drawer':
                    drawer_name = obj_name
        # Fallback names
        if drawer_name is None:
            drawer_name = 'drawer1'
        if handle_name is None:
            handle_name = 'handle1'

        # 3) Try to pull (open) the drawer if possible
        try:
            # To pull: must hold the handle, robot at location, drawer unlocked and closed
            print(f"[Plan] Trying to open {drawer_name} by pulling {handle_name} at {current_loc}...")
            obs, reward, done = execute_pick(env, task, handle_name, current_loc)
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, current_loc)
        except Exception as e:
            print(f"[Plan] Error opening drawer: {str(e)}")

        # 4) Identify an object on the floor to pick up
        object_to_pick = None
        for obj_name, obj_info in positions.items():
            if 'type' in obj_info and obj_info['type'] == 'object':
                object_to_pick = obj_name
                break
        if object_to_pick is None:
            object_to_pick = 'item1'  # fallback

        # 5) Pick the object
        try:
            print(f"[Plan] Picking up {object_to_pick} at {current_loc}...")
            obs, reward, done = execute_pick(env, task, object_to_pick, current_loc)
        except Exception as e:
            print(f"[Plan] Error picking object: {str(e)}")

        # 6) Place the object into the drawer
        try:
            print(f"[Plan] Placing {object_to_pick} into {drawer_name} at {current_loc}...")
            obs, reward, done = execute_place(env, task, object_to_pick, drawer_name, current_loc)
        except Exception as e:
            print(f"[Plan] Error placing object: {str(e)}")

        # 7) Optionally, close the drawer
        try:
            print(f"[Plan] Closing {drawer_name} at {current_loc}...")
            obs, reward, done = execute_push(env, task, drawer_name, current_loc)
        except Exception as e:
            print(f"[Plan] Error closing drawer: {str(e)}")

        print("[Plan] Task (plan) execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
