# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use the predefined primitive/exploration skills directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with exploration phase for missing predicate.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # This should give a dictionary like {'drawer': (x,y,z), 'handle': (x,y,z), ...}
        positions = get_object_positions()

        # -- STEP 1: EXPLORATION PHASE -- #
        #
        # Feedback: (robot-at drawer-area)  --> Implies system is missing knowledge of
        # what "drawer-area" maps to in state, or if the predicate "robot-at" is satisfied.
        #
        # Use 'execute_go' (primitive), or if using exploration domain use 'execute_go_identify'
        # or similar skills as available. Here, using primitives only as per skills list.
        #
        # We attempt to move to 'drawer-area'. If skill fails or predicate is missing, capture the exception.
        #
        # Assumption: Positions/map keys will contain a location corresponding to the drawer area.

        # Try finding the location key for the drawer-area
        drawer_area_key_candidates = [k for k in positions.keys() if 'drawer' in k or 'area' in k]
        if drawer_area_key_candidates:
            drawer_area_key = drawer_area_key_candidates[0]
        else:
            drawer_area_key = None

        # Just pick the first available as fallback, but log error if not found.
        if drawer_area_key is None:
            print("[Exploration] Could not find a plausible key for drawer-area in positions!")
            print("[Exploration] Positions:", positions)
        else:
            print(f"[Exploration] Using area key '{drawer_area_key}' for drawer-area.")

        # Get current robot location (requires observation fields)
        current_robot_location = None
        if 'robot' in positions:
            current_robot_location = positions['robot']
        # Or infer from observation/state representation (if available)
        # Placeholder: if observation or state provides robot current location, extract here.

        # Assume that location names are string, or if positions are coordinates, the skill interface will map names.
        # We will use skill: execute_go(from_location, to_location)
        #
        # As we do not have the exact observation format or plan, we will check if the move is possible.
        # If robot is not at drawer-area, try to move there.
        try:
            if drawer_area_key is not None:
                # If you need location names instead of positions, use drawer_area_key as the identifier.
                print(f"[Exploration] Attempting to move to {drawer_area_key} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=None,    # Typically you would pass actual current location if you can get it
                    to_location=drawer_area_key
                )
                print("[Exploration] Moved robot to drawer-area.")
            else:
                print("[Exploration] Skipping move to drawer-area; key not found.")
        except Exception as e:
            print(f"[Exploration] Exception during move to drawer-area: {str(e)}")
            print("[Exploration] This may indicate a missing predicate such as (robot-at drawer-area) or mapping issue.")
            # This is the missing predicate inference step per instruction.
            # You could log this or trigger a further exploration skill here if available.

        # -- STEP 2: PLAN/GOAL PHASE (replace with your oracle plan logic) -- #
        #
        # Here is where you would execute the rest of your oracle plan, step by step,
        # using only the available skills such as execute_pick, execute_place, etc.
        #
        # A sample placeholder sequence using only predefined skills follows:

        # The following is a generic structure. Replace objects/locations with
        # the actual ones from your task.
        try:
            # Example: Pick up an object from floor at drawer-area
            object_key_candidates = [k for k in positions.keys() if k != drawer_area_key and 'object' in k]
            if object_key_candidates:
                target_object_key = object_key_candidates[0]
                print(f"[Plan] Attempting to pick up object '{target_object_key}' at '{drawer_area_key}'.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=target_object_key,
                    p=drawer_area_key
                )
                print("[Plan] Picked up object.")
            else:
                print("[Plan] No object key found to pick up.")

            # Example: Pull to unlatch/open drawer (requires handle-of, etc.)
            # Find a handle key if available
            handle_key_candidates = [k for k in positions.keys() if 'handle' in k]
            if handle_key_candidates:
                handle_key = handle_key_candidates[0]
                print(f"[Plan] Attempting to pull handle '{handle_key}' at '{drawer_area_key}'.")
                # Find drawer key for the handle
                drawer_key_candidates = [k for k in positions.keys() if 'drawer' in k]
                if drawer_key_candidates:
                    drawer_key = drawer_key_candidates[0]
                    try:
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            d=drawer_key,
                            h=handle_key,
                            p=drawer_area_key
                        )
                        print("[Plan] Pulled handle and opened drawer.")
                    except Exception as e:
                        print(f"[Plan] Exception during pull handle: {str(e)}")
                else:
                    print("[Plan] No drawer key found for pulling action.")
            else:
                print("[Plan] No handle found for pull.")

            # Example: Place object into drawer
            if object_key_candidates and drawer_key_candidates:
                print(f"[Plan] Attempting to place '{target_object_key}' into drawer '{drawer_key}' at '{drawer_area_key}'.")
                obs, reward, done = execute_place(
                    env,
                    task,
                    o=target_object_key,
                    d=drawer_key,
                    p=drawer_area_key
                )
                print("[Plan] Placed object in drawer.")
            else:
                print("[Plan] Place action skipped; missing object or drawer.")

            # Example: Push to close drawer
            if drawer_key_candidates:
                print(f"[Plan] Attempting to push (close) drawer '{drawer_key}' at '{drawer_area_key}'.")
                obs, reward, done = execute_push(
                    env,
                    task,
                    d=drawer_key,
                    p=drawer_area_key
                )
                print("[Plan] Pushed drawer closed.")
            else:
                print("[Plan] No drawer key found for push.")

        except Exception as e:
            print(f"[Plan] Exception in plan execution: {str(e)}")

        # -- END OF PLAN --

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()