# run_skeleton_task.py (Completed With Exploration and Predicate Identification)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # use all primitive skills directly as per instructions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Check Predicate (e.g. drawer-unlocked) ---
        # The feedback suggests we need to determine the "drawer-unlocked" status.

        # We'll use exploration logic to "discover" whether drawer1 is unlocked.
        # This typically means attempting to interact with the drawer handle, and observing the effect.

        # For demonstration, let's assume the following convention for objects:
        #   - 'drawer1': the drawer object
        #   - 'drawer1_handle': the handle object attached to drawer1
        #   - 'robot': the robot
        #   - 'locations': a dict of location name to coordinates

        # We'll use these names safely, but in a real setup you'd ensure consistency with env/task.
        try:
            drawer_name = 'drawer1'
            handle_name = 'drawer1_handle'
            robot_location = positions['robot']
            handle_location = positions[handle_name]
            drawer_location = positions[drawer_name]
        except Exception as e:
            print("ERROR: Could not find objects in position dict:", str(e))
            shutdown_environment(env)
            return

        # We want to identify if (drawer-unlocked drawer1) holds.
        # The exploration domain allows us to try pulling the handle and check for lock-known property.
        # Since we can only use provided skills, we perform interactions via skill functions.

        print("[Exploration] Moving to handle of the drawer to check lock state...")

        # 1. Move to the drawer or handle location:
        try:
            obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=handle_location)
            if done:
                print("[Exploration] Finished after moving to handle.")
                return
            print("[Exploration] Moved to handle.")
        except Exception as e:
            print(f"[Exploration] WARNING: Could not move to handle: {e}")

        # 2. Attempt to pick the handle if required in the plan.
        try:
            obs, reward, done = execute_pick(env, task, object=handle_name, location=handle_location)
            if done:
                print("[Exploration] Could not pick handle (maybe already holding or not at location), continuing...")
        except Exception as e:
            print(f"[Exploration] WARNING: Could not pick handle: {e}")

        # 3. Attempt to "pull" the drawer (effectively will check 'drawer-unlocked' predicate, since pull is only successful if drawer is unlocked)
        drawer_unlocked = False
        try:
            obs, reward, done = execute_pull(env, task, drawer=drawer_name, handle=handle_name, location=handle_location)
            if done:
                drawer_unlocked = True
                print("[Exploration] Drawer pull succeeded: drawer is unlocked.")
            else:
                print("[Exploration] Drawer pull did not complete task, but may still be unlocked.")
        except Exception as e:
            # If we cannot pull, maybe drawer is locked
            print(f"[Exploration] WARNING: Could not pull drawer: {e}")
            print("[Exploration] Assuming drawer is locked or some other precondition failed.")

        # (In a real scenario we might query the environment state or deduce based on exception/observation.)
        # For now, based on feedback, suppose we learned the "drawer-unlocked" status.

        # --- Main Task Plan (using only primitive skill functions) ---

        # { The oracle plan would be executed here, for demonstration let's simulate an archetypal plan:
        #    (1) Move to the drawer location
        #    (2) Pull drawer to open
        #    (3) Move to object location
        #    (4) Pick object (e.g. pick up pen on the floor)
        #    (5) Move back to drawer
        #    (6) Place object inside drawer
        #    (7) Push the drawer to close it
        # }

        # 1. Move to drawer location (if not already there)
        try:
            obs, reward, done = execute_go(env, task, from_location=handle_location, to_location=drawer_location)
            if done:
                print("[Task] Arrived at drawer.")
        except Exception as e:
            print(f"[Task] Could not move to drawer location: {e}")

        # 2. Ensure drawer is open (may have been opened above, but if not, open it now)
        try:
            # We may need to pick handle again if not holding it already.
            obs, reward, done = execute_pick(env, task, object=handle_name, location=drawer_location)
        except Exception:
            pass  # Already holding or failed: try to pull anyway

        try:
            obs, reward, done = execute_pull(env, task, drawer=drawer_name, handle=handle_name, location=drawer_location)
            print("[Task] Drawer opened.")
        except Exception as e:
            print("[Task] Could not pull open the drawer:", e)

        # 3. Move to object location (find 'target_obj' on the floor)
        try:
            target_obj = None
            target_location = None
            # Find any object that is 'on-floor'
            for obj, pos in positions.items():
                if obj.startswith('obj') or obj.startswith('item') or obj == 'pen':  # adapt as needed
                    target_obj = obj
                    target_location = pos
                    break
            if target_obj is None:
                print("[Task] No object to pick, aborting plan.")
                shutdown_environment(env)
                return
            obs, reward, done = execute_go(env, task, from_location=drawer_location, to_location=target_location)
            print(f"[Task] Going to pick up object {target_obj}.")
        except Exception as e:
            print(f"[Task] Could not move to object location: {e}")

        # 4. Pick up the object
        try:
            obs, reward, done = execute_pick(env, task, object=target_obj, location=target_location)
            print(f"[Task] Picked up {target_obj}")
        except Exception as e:
            print(f"[Task] Could not pick up object: {e}")

        # 5. Move back to drawer (to place the object in drawer)
        try:
            obs, reward, done = execute_go(env, task, from_location=target_location, to_location=drawer_location)
            print("[Task] Returned to drawer with object.")
        except Exception as e:
            print(f"[Task] Could not return to drawer: {e}")

        # 6. Place object inside the drawer
        try:
            obs, reward, done = execute_place(env, task, object=target_obj, drawer=drawer_name, location=drawer_location)
            print(f"[Task] Placed {target_obj} inside {drawer_name}.")
        except Exception as e:
            print(f"[Task] Could not place object in drawer: {e}")

        # 7. Push (close) the drawer
        try:
            obs, reward, done = execute_push(env, task, drawer=drawer_name, location=drawer_location)
            print(f"[Task] Closed {drawer_name}. Task Complete.")
        except Exception as e:
            print(f"[Task] Could not close drawer: {e}")

        # If more objects and handling are needed, extend the plan accordingly.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()