# run_skeleton_task.py (Filled for Predicate Exploration: drawer-closed)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use pre-defined skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # == Exploration Phase: Find the missing predicate (from feedback) ==
        #
        # Based on feedback, we need to explore for the presence of predicate "drawer-closed".
        # The missing predicate is typically discovered via interaction (e.g., try to open or close a drawer and inspect its state).
        #
        # Let's iterate over known drawers and handles, and use available skills to examine closing and opening
        # and check if the appropriate state changes are observed.

        # Let's assume positions includes: 'drawers', 'handles', and 'locations'
        drawers = positions.get('drawers', [])
        handles = positions.get('handles', [])
        robot_location = positions.get('robot', None)
        all_locations = positions.get('locations', [])

        # Attempt to find a drawer and handle to interact with
        if not drawers or not handles or robot_location is None:
            print("[Exploration] No drawers, handles, or robot location found in environment.")
        else:
            # We'll assume one robot, operate on first accessible drawer/handle
            drawer = drawers[0]
            handle = None
            for h in handles:
                # Assume handle has key 'drawer' for mapping
                if h.get('drawer', None) == drawer['name']:
                    handle = h
                    break
            if not handle:
                handle = handles[0]  # Fallback: just pick first handle

            # Find where drawer is located (could be its own field or attached to a location)
            drawer_location = drawer.get('location', robot_location)

            # Move the robot to the drawer location if needed
            if robot_location != drawer_location:
                try:
                    print(f"[Exploration] Moving robot to location of drawer: {drawer_location}")
                    obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                    robot_location = drawer_location
                except Exception as e:
                    print(f"[Exploration] Failed to execute_go: {e}")
                    return

            # Ensure robot's hand is empty and free before picking
            try:
                print(f"[Exploration] Executing gripper reset to free hand")
                obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print(f"[Exploration] execute_gripper failed: {e}")
            
            # Pick the handle for the drawer
            try:
                print(f"[Exploration] Trying to pick handle '{handle['name']}' at location '{drawer_location}'")
                obs, reward, done = execute_pick(env, task, handle['name'], drawer_location)
                if done:
                    print("[Exploration] Picked the handle.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

            # Try pulling (open drawer)
            try:
                print(f"[Exploration] Try to open (pull) drawer '{drawer['name']}' with handle '{handle['name']}'")
                obs, reward, done = execute_pull(env, task, drawer['name'], handle['name'], drawer_location)
                if done:
                    print(f"[Exploration] Drawer '{drawer['name']}' opened.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

            # Try pushing (close drawer) and see if the predicate for "drawer-closed" is used/affected
            try:
                print(f"[Exploration] Try to close (push) drawer '{drawer['name']}' at location '{drawer_location}'")
                obs, reward, done = execute_push(env, task, drawer['name'], drawer_location)
                if done:
                    print(f"[Exploration] Drawer '{drawer['name']}' closed (exploring for 'drawer-closed' predicate).")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # -- End Exploration Phase --

        print("[Exploration] Finished routine for exploring missing predicate 'drawer-closed' using available skills.")

        # Optionally: Continue with main oracle plan or further exploration/actions

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
