# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use externally-defined skill primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object locations and environmental information
        positions = get_object_positions()

        # ===========================
        # === Exploration Phase ===
        # ===========================
        #
        # Exploration objective:
        # Detect if the predicate (drawer-unlocked drawer1) is present or missing.
        # Use exploration knowledge/skills to actively determine the lock state.
        #

        # Example: Assume the robot and drawer handle are named as follows.
        # (Modify names/keys as appropriate for your environment/configuration.)
        robot = 'robot1' if 'robot1' in positions else list(positions.keys())[0]
        drawer = 'drawer1'
        handle = 'handle1' if 'handle1' in positions else None

        # Determine suitable locations for robot and drawer
        drawer_pos = positions.get(drawer, None)
        handle_pos = positions.get(handle, None)
        all_locations = [k for k, v in positions.items() if 'location' in k or 'loc' in k]
        if not all_locations:
            all_locations = [drawer]  # fallback

        # Exploration: Try to pull the drawer's handle to check its lock state
        # We call 'execute_pull' if the robot can get to the handle and grasp it.
        exploration_success = False
        missing_predicate_found = False

        try:
            if handle and handle_pos is not None:
                # Move robot to handle location (if not already there)
                robot_loc = None
                for k, v in positions.items():
                    if k == robot:
                        robot_loc = v
                        break
                if robot_loc != handle_pos:
                    # Use execute_go to move robot
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=robot_loc,
                        to_location=handle_pos,
                        max_steps=100
                    )
                # Pick the handle
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object=handle,
                    location=handle_pos,
                    max_steps=100
                )
                # Try to pull the handle to open the drawer
                obs, reward, done = execute_pull(
                    env,
                    task,
                    d=drawer,
                    h=handle,
                    p=handle_pos,
                    max_steps=100
                )
                exploration_success = True
            else:
                print("[Exploration] Handle object or its position is missing; skipping exploration phase.")

        except Exception as ex:
            # If action failed, it's likely due to missing preconditions (e.g., drawer-unlocked not present),
            # which would signal that the missing predicate is (drawer-unlocked drawer1)
            print(f"[Exploration] Exception encountered during execute_pull/check: {ex}")
            print("[Exploration] Inferring that (drawer-unlocked drawer1) is likely the missing predicate.")
            missing_predicate_found = True

        # After exploration, you could log or flag the discovered predicate
        if missing_predicate_found:
            print(">>> [Exploration] MISSING PREDICATE DETECTED: (drawer-unlocked drawer1) <<<")

        # =================================
        # === Example Task Plan (Generic) ===
        # =================================

        # For demonstration, simulate the main task. 
        # Insert plan steps here using only available skills if desired.
        # Example: Move to drawer, open it, and place an object

        # -- Example Execution (adapt to your setup if needed) --

        # 1. Move robot to drawer location (assume robot is not already there)
        try:
            robot_loc = None
            for k, v in positions.items():
                if k == robot:
                    robot_loc = v
                    break
            if drawer_pos is not None and robot_loc != drawer_pos:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_loc,
                    to_location=drawer_pos,
                    max_steps=100
                )
        except Exception as ex:
            print(f"[Task] Error moving robot to drawer: {ex}")

        # 2. Try to pull handle again (assuming drawer is now unlocked, if needed)
        try:
            if handle and handle_pos is not None:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object=handle,
                    location=handle_pos,
                    max_steps=100
                )
                obs, reward, done = execute_pull(
                    env,
                    task,
                    d=drawer,
                    h=handle,
                    p=handle_pos,
                    max_steps=100
                )
                print("[Task] Pulled drawer to open.")
        except Exception as ex:
            print(f"[Task] Error pulling handle: {ex}")

        # 3. Pick object from floor (find any object that's on-floor in positions)
        picked_object = None
        for obj, pos in positions.items():
            if "object" in obj and pos is not None:
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object=obj,
                        location=pos,
                        max_steps=100
                    )
                    picked_object = obj
                    print(f"[Task] Picked up object {obj}.")
                    break
                except Exception as ex:
                    continue

        # 4. Place object in drawer (if picked)
        try:
            if picked_object and drawer_pos is not None:
                obs, reward, done = execute_place(
                    env,
                    task,
                    o=picked_object,
                    d=drawer,
                    p=drawer_pos,
                    max_steps=100
                )
                print(f"[Task] Placed object {picked_object} in {drawer}.")
        except Exception as ex:
            print(f"[Task] Error placing object: {ex}")

        # 5. Push the drawer to close it
        try:
            if drawer_pos is not None:
                obs, reward, done = execute_push(
                    env,
                    task,
                    d=drawer,
                    p=drawer_pos,
                    max_steps=100
                )
                print("[Task] Closed the drawer.")
        except Exception as ex:
            print(f"[Task] Error closing drawer: {ex}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()