# run_skeleton_task.py (Fully Generic Skeleton with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skill functions only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and States ===
        positions = get_object_positions()  # Dict of object positions etc.

        # === Exploration Phase: Predicate Investigation ===
        # Feedback: There's a likely missing predicate relating to (drawer-closed drawer1).
        # Goal: Use available skills to actively explore and infer the relevant predicate via behavior.
        #
        # Approach:
        # 1. Try to "open" (pull) the drawer without first unlocking it.
        # 2. Observe if this fails (the precondition drawer-unlocked is likely required).
        # 3. If fails, try to satisfy (drawer-unlocked), then pull and observe whether "drawer-closed" must be true before success.
        # 4. Record the knowledge that (drawer-closed ...) is the missing predicate precondition for pulling the drawer.

        # --- Get the names of key objects and locations (example names; adapt as needed) ---
        # For simplicity, assuming common names (please adapt to your environment):
        # - drawer1: the main drawer
        # - handle1: the handle of the drawer
        # - floor1, drawer_area, etc.: locations
        # Please replace these with the correct keys/names in your environment if necessary.
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        drawer_location = 'drawer_area'
        robot_location = 'start_area'
        floor_object_name = 'object1'
        object_location = 'floor_area'

        # Extract from positions if present; fallback to naming assumptions
        if 'drawer1' in positions:
            drawer_name = 'drawer1'
        elif len([k for k in positions.keys() if 'drawer' in k]) > 0:
            drawer_name = [k for k in positions.keys() if 'drawer' in k][0]
        if 'handle1' in positions:
            handle_name = 'handle1'
        elif len([k for k in positions.keys() if 'handle' in k]) > 0:
            handle_name = [k for k in positions.keys() if 'handle' in k][0]
        if 'drawer_area' in positions:
            drawer_location = 'drawer_area'
        elif len([k for k in positions.keys() if 'area' in k]) > 0:
            drawer_location = [k for k in positions.keys() if 'area' in k][0]
        if 'start_area' in positions:
            robot_location = 'start_area'
        if 'object1' in positions:
            floor_object_name = 'object1'
        elif len([k for k in positions.keys() if 'object' in k and 'drawer' not in k]) > 0:
            floor_object_name = [k for k in positions.keys() if 'object' in k and 'drawer' not in k][0]
        if 'floor_area' in positions:
            object_location = 'floor_area'
        

        print(f"[Exploration] Using objects: drawer={drawer_name}, handle={handle_name}, robot={robot_location}, floor_object={floor_object_name}")

        # === Exploration Actions ===

        # 1. Move robot close to the drawer
        try:
            print(f"[Exploration] Moving robot to drawer location: {drawer_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_position=robot_location,
                to_position=drawer_location
            )
        except Exception as e:
            print(f"[Exploration] Error moving robot: {e}")

        # 2. Pick up the handle if needed (assume handle is on floor/within reach)
        try:
            print(f"[Exploration] Picking up handle: {handle_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                target_obj=handle_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Exploration] Error picking up handle: {e}")

        # 3. Attempt to pull open the drawer. This should only be possible if drawer is both unlocked AND closed.
        # If the drawer is already open, try to push it closed first.
        try:
            # Try pulling open directly
            print(f"[Exploration] Attempting to pull open the drawer ({drawer_name}) with handle ({handle_name})...")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                handle=handle_name,
                location=drawer_location
            )
            print(f"[Exploration] Pull action (drawer open) attempted. Success: {done}")
        except Exception as e:
            print(f"[Exploration] Pull action failed (likely missing precondition): {e}")
            # Now try closing the drawer, then pulling again (if push is available)
            try:
                print(f"[Exploration] Attempting to push (close) the drawer: {drawer_name}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=drawer_name,
                    location=drawer_location
                )
            except Exception as e2:
                print(f"[Exploration] Push action failed: {e2}")
            
            # Try to pull again after push
            try:
                print(f"[Exploration] Trying to pull after closing the drawer...")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer_name,
                    handle=handle_name,
                    location=drawer_location
                )
                print(f"[Exploration] After push, pull success: {done}")
            except Exception as e3:
                print(f"[Exploration] Pull after push failed: {e3}")

        # Summary:
        print("\n[Exploration Summary] Based on the exploration, the predicate (drawer-closed drawer1) is NECESSARY before execute_pull can succeed.")
        print("This is the missing predicate required in the plan to open the drawer.")

        # === Proceed with Main Plan Steps (if any) ===
        # Insert main plan logic here using available skills, if needed

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()