# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided primitives directly

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicate.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Positions is assumed to be a dictionary: e.g., {'floor': (x,y,z), 'drawer': (x,y,z), ...}
        positions = get_object_positions()

        # Exploration phase: try to detect the missing predicate using exploration actions
        print("[Exploration] Checking for missing predicates in world model...")

        # --- Knowledge: Feedback indicates (robot-at floor)
        # If (robot-at floor) is not recognized, exploration attempts to discover how robot-at/at predicates are constructed.

        # We stub out robot/locations/object names accordingly (may need to adjust as per your env and feedback)
        # Let's attempt to perform moves and sense, to see if (robot-at floor) corresponds to correct grounding.

        try:
            # Step 1: Ensure robot is free to move
            # Step 2: Attempt to move the robot to "floor" location, using the available skill execute_go
            from_loc = None
            to_loc = "floor"
            # Find the robot's current location
            # Many environments include this in their initial obs, but here positions may help
            
            # Find a current location for the robot from object_positions keys (naive search)
            # Assume keys like 'floor', 'drawer_area', etc.
            robot_locs = [k for k in positions.keys() if 'floor' in k or 'area' in k or 'loc' in k]
            if len(robot_locs) == 0:
                robot_locs = list(positions.keys())
            # Try first as from location
            if len(robot_locs) == 0:
                raise RuntimeError("Cannot infer robot initial location from positions.")
            from_loc = robot_locs[0]
            if to_loc not in positions:
                to_loc = from_loc  # fallback, in case no distinct 'floor'
            
            print(f"[Exploration] Attempting to move robot from '{from_loc}' to '{to_loc}' using execute_go.")
            obs, reward, done = execute_go(
                env, task, from_location=from_loc, to_location=to_loc
            )
            print("[Exploration] Movement executed. Checking post conditions.")

        except Exception as ex:
            print(f"[Exploration] Error during execute_go: {ex}")

        # Try to pick an object on the floor using execute_pick
        # This will help check if predicates such as (on-floor ...) and (robot-at ...) are recognized appropriately
        try:
            floor_objs = [k for k in positions.keys() if k.startswith('obj') or k.startswith('object') or 'ball' in k]
            if len(floor_objs) > 0:
                obj_to_pick = floor_objs[0]
                pick_pos = positions[obj_to_pick]
                print(f"[Exploration] Attempting to pick '{obj_to_pick}' at '{to_loc}'.")
                obs, reward, done = execute_pick(
                    env, task, object_name=obj_to_pick, location=to_loc
                )
                print("[Exploration] Pick executed. Checking outcome.")
        except Exception as ex:
            print(f"[Exploration] Error during execute_pick: {ex}")

        # Additional: You can try using execute_sweep or execute_gripper if those make sense for the exploration

        # === Main Task Plan (as per logical flow and available knowledge) ===
        # Insert your oracle plan execution steps here!
        # For example, to open a drawer and place an object:
        #
        # 1. Move to location
        # 2. Pick object from floor
        # 3. Pull drawer
        # 4. Place object in drawer
        # 5. Push drawer
        #
        # However, as the actual oracle plan is not given in the input, we keep placeholders.

        # Example sequence making maximum use of provided skill names:

        try:
            # 1. Move robot to "floor" area
            print("[Plan] Moving robot to 'floor' area.")
            obs, reward, done = execute_go(env, task, from_location=from_loc, to_location='floor')

            # 2. Pick object if any exists on floor
            if len(floor_objs) > 0:
                print(f"[Plan] Picking up object '{obj_to_pick}' from 'floor'.")
                obs, reward, done = execute_pick(env, task, object_name=obj_to_pick, location='floor')

            # 3. If there is a drawer and handle, attempt to open it
            drawers = [k for k in positions.keys() if 'drawer' in k]
            handles = [k for k in positions.keys() if 'handle' in k or 'knob' in k]
            if len(drawers) > 0 and len(handles) > 0:
                drawer_name = drawers[0]
                handle_name = handles[0]
                print(f"[Plan] Pulling open the drawer '{drawer_name}' with handle '{handle_name}'.")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, 'floor')  # assuming location 'floor'
            
            # 4. Place the object in the drawer if possible
            if len(drawers) > 0 and len(floor_objs) > 0:
                drawer_name = drawers[0]
                print(f"[Plan] Placing object '{obj_to_pick}' into drawer '{drawer_name}'.")
                obs, reward, done = execute_place(env, task, object_name=obj_to_pick, drawer_name=drawer_name, location='floor')
            
            # 5. Push/close the drawer
            if len(drawers) > 0:
                drawer_name = drawers[0]
                print(f"[Plan] Pushing closed the drawer '{drawer_name}'.")
                obs, reward, done = execute_push(env, task, drawer_name, 'floor')

        except Exception as ex_plan:
            print(f"[Plan] Exception during main plan execution: {ex_plan}")

        print("[Task] Exploration and main plan steps finished. Task run is now complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()