# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # (All skills are imported here; do not redefine.)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Run a skeleton exploration to find missing predicates and test skill coverage.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Get Object Positions ===
        positions = get_object_positions()

        # === Skill/Predicate Exploration for Debugging/Diagnosis ===
        # Because previous planning timed out (possibly due to missing predicates), 
        # we conduct an exploration phase using available skills, attempting to discover
        # what knowledge/predicate is missing in the domain/problem definition.
        #
        # In this context, the missing predicate most likely refers to object properties
        # or states (e.g., lock-known, identified), as in the exploration knowledge model.
        
        # List of available skills
        available_skills = [
            'execute_pick', 
            'execute_place', 
            'execute_push', 
            'execute_pull', 
            'execute_sweep', 
            'execute_rotate', 
            'execute_go', 
            'execute_gripper'
        ]

        # Extract object/location names if possible
        try:
            object_names = [k for k in positions.keys() if not k.startswith('robot') and not 'drawer' in k and not 'handle' in k]
            drawer_names = [k for k in positions.keys() if 'drawer' in k]
            handle_names = [k for k in positions.keys() if 'handle' in k]
            location_names = [k for k in positions.keys() if ("loc" in k or "room" in k or "area" in k)]
            # Fallback to generic "location" if no named locations.
            if not location_names and len(positions) > 0:
                location_names = list(positions.keys())
        except Exception as e:
            print("Error extracting object/location names from positions dict:", str(e))
            object_names, drawer_names, handle_names, location_names = [], [], [], ""

        # Step 1: Exploration -- Try all basic motion/observation skills to stimulate predicate discovery
        # The goal: See if something "fails" or is unsatisfiable (e.g., unexplained locked state), 
        # which would point to a missing knowledge/predicate (as per feedback).

        # -- 1. Try going to each location --
        for from_location in location_names:
            for to_location in location_names:
                if from_location == to_location:
                    continue
                try:
                    print(f"[Exploration] Attempting execute_go: {from_location}->{to_location}")
                    obs, reward, done, info = execute_go(env, task, from_location, to_location)
                    if done:
                        print("[Exploration] Task signaled done after execute_go.")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_go failed: {from_location}->{to_location} | {str(e)}")

        # -- 2. Try picking, pulling, pushing on objects/drawers to trigger any missing lock, type, or state predicate --
        for obj in object_names:
            # Find a plausible location for the object (from positions or default)
            obj_pos = positions.get(obj, None)
            # Use first available location, else None.
            for loc in location_names:
                try:
                    # -- Try pick --
                    print(f"[Exploration] Attempting execute_pick: object={obj}, location={loc}")
                    obs, reward, done, info = execute_pick(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended after execute_pick.")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {obj}@{loc} | {str(e)}")

        # -- 3. Try pulling handles to test lock-known/identified-like predicate --
        for drawer in drawer_names:
            # Find handle associated with drawer (if possible)
            drawer_handle = None
            for handle in handle_names:
                if handle.startswith('handle') and drawer in handle:
                    drawer_handle = handle
                    break
            if not drawer_handle and handle_names:
                drawer_handle = handle_names[0]
            # Use first location as location
            loc = location_names[0] if location_names else None
            if drawer_handle and loc:
                try:
                    print(f"[Exploration] Attempting execute_pull: drawer={drawer}, handle={drawer_handle}, location={loc}")
                    obs, reward, done, info = execute_pull(env, task, drawer, drawer_handle, loc)
                    if done:
                        print("[Exploration] Task finished after execute_pull.")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pull failed: {drawer}, {drawer_handle}, {loc} | {str(e)}")

        # -- 4. Try placing objects into drawers (to potentially stimulate unknown state predicates) --
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting execute_place: object={obj}, drawer={drawer}, location={loc}")
                        obs, reward, done, info = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print("[Exploration] Task ended after execute_place.")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {obj}, {drawer}, {loc} | {str(e)}")

        # -- 5. Try sweep (as an exploratory manipulation) --
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting execute_sweep: object={obj}, location={loc}")
                    obs, reward, done, info = execute_sweep(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended after execute_sweep.")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {obj}, {loc} | {str(e)}")

        # -- 6. Try gripper manipulation (robot-free prerequisite only) --
        try:
            print(f"[Exploration] Attempting execute_gripper.")
            obs, reward, done, info = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended after execute_gripper.")
                return
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {str(e)}")

        # -- 7. Optional: Try execute_rotate if implementation available --
        try:
            print(f"[Exploration] Attempting execute_rotate (if implemented).")
            obs, reward, done, info = execute_rotate(env, task)
            if done:
                print("[Exploration] Task ended after execute_rotate.")
                return
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {str(e)}")

        print("===== Exploration complete. Check the logs above for evidence of actions that could not be performed due to missing information or predicates. These may indicate the required predicate to add or observe in the planning domain/problem. =====")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
