# run_skeleton_task.py (Completed Execution Code for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Available: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

from video import init_video_writers, recording_step, recording_get_observation

# Provides object positions/state info in the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # -- Begin Exploration Phase (To Discover Missing Predicate) --

        # Based on the feedback "(drawer-open drawer3)", the robot should perform exploration to detect which drawer(s) are open.
        # We'll assume that among the available drawers, 'drawer3' is referenced, so we need to check the status for 'drawer3'.

        # For the exploration, we must check for all known drawers in the environment; sample exploration is generic.

        # List of candidate drawer names, adapt if you have a separate way to retrieve drawer IDs
        drawer_names = [name for name in positions.keys() if "drawer" in name]

        found_open_drawers = []

        for drawer in drawer_names:
            try:
                print(f"[Exploration] Attempting to check open status for: {drawer}")
                # Assuming the robot's gripper/effector must be at drawer location to check/open it
                # Retrieve position for the drawer
                drawer_pos = positions[drawer]

                # Get robot's current location (assume accessible as 'robot' key, else define initial loc)
                # (Adapt this appropriately if your env/task gives different naming)
                robot_loc = None
                for k in positions:
                    if 'robot' in k:
                        robot_loc = positions[k]
                        break

                # If current robot loc != drawer loc, navigate ("execute_go")
                # For the sake of using available actions and as per domain, 'execute_go' takes from, to as location names
                # Here we use the key names as location identifiers
                try:
                    execute_go(env, task, from_location='robot_home', to_location=drawer)
                except Exception as e:
                    print(f"[Exploration] Move to {drawer} failed or not needed: {e}")

                # Now execute a "pull" attempt on the drawer to see if it's open/unlocked (skill: execute_pull)
                # For execute_pull: expects (drawer, handle-object, location)
                # Need handle-object: scan for handle associated with this drawer
                # We'll search in positions for a handle related to this drawer
                handle_obj = None
                for obj in positions.keys():
                    if "handle" in obj and drawer in obj:
                        handle_obj = obj
                        break

                # If not found, try generic 'handle'
                if handle_obj is None:
                    for obj in positions.keys():
                        if "handle" in obj:
                            handle_obj = obj
                            break

                if handle_obj is not None:
                    try:
                        # To pull: must first "pick" the handle-object if needed (precondition: holding the handle-object)
                        # Ensure robot hand is empty first (could execute execute_gripper())
                        try:
                            execute_gripper(env, task)
                        except Exception:
                            pass
                        # Attempt to pick up the handle-object
                        execute_pick(env, task, object_name=handle_obj, location_name=drawer)
                        # Now try pulling the drawer using the handle
                        execute_pull(env, task, drawer_name=drawer, handle_object_name=handle_obj, location_name=drawer)
                        # After pulling, if no error, mark as open
                        print(f"[Exploration] {drawer} appears OPEN after pull attempt.")
                        found_open_drawers.append(drawer)
                    except Exception as pull_e:
                        print(f"[Exploration] {drawer}: Pull attempt failed or not open/unlocked: {pull_e}")

                else:
                    print(f"[Exploration] No handle found for {drawer}, skipping pull.")

            except Exception as e:
                print(f"[Exploration] Exploration of {drawer} failed: {e}")

        if found_open_drawers:
            print(f"[Exploration] Discovered open drawers: {found_open_drawers}")
        else:
            print("[Exploration] No open drawers found or could not confirm open status.")

        # -- End of Exploration Phase --

        # Next, oracle plan can be executed using found information.
        # Here, based on PDDL/feedback, the missing predicate was (drawer-open drawer3).
        # You may now use this knowledge for the remainder of your higher-level plan.

        # TODO: Further plan steps would go here using only the provided skill functions.
        # For demonstration, code executes only the exploration to discover open drawers.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
