# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all pre-defined skill functions as provided

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Positions: dictionary keyed by object/landmark names
        positions = get_object_positions()

        # ========================
        # === Exploration Phase ==
        # ========================
        #
        # Purpose: Discover which predicate or world fact is "missing" or not established,
        #          by using only available exploration actions and predefined skills.
        #
        # - Use available exploration-oriented skills:
        #   ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # - Focus on those related to information gathering and interaction.
        #   (from the exploration domain: e.g., execute_pull enables discovery of lock-known, and so on)
        #
        # - Here, we programmatically try out available actions related to exploration,
        #   attempt to observe their outcome, and log any accessible facts to help
        #   the user spot which predicates are missing.

        print("[Exploration] Starting systematic exploration for missing predicates.")

        # We'll iterate over objects and locations, and attempt basic exploration actions.
        # Collect the names for objects, drawers, handles, and locations from the environment.
        # These names/types may need to be retrieved or inferred from observation or positions.

        # Extract object names/types from observation or positions
        object_names = []
        drawer_names = []
        handle_names = []
        location_names = []

        # Classify entities by some heuristics found in PDDL naming conventions and positions keys
        for k in positions.keys():
            if 'drawer' in k:
                drawer_names.append(k)
            elif 'handle' in k:
                handle_names.append(k)
            elif 'loc' in k or 'place' in k or 'room' in k:
                location_names.append(k)
            else:
                object_names.append(k)

        # Sometimes we have a dedicated list of locations, fallback to unique positions if not
        if not location_names:
            # Just use all unique non-object, non-drawer positions as location proxies
            location_names = list(set(positions.values()))

        # For the following, we operate only if type lists are non-empty
        print(f"[Exploration] Found objects: {object_names}")
        print(f"[Exploration] Found drawers: {drawer_names}")
        print(f"[Exploration] Found handles: {handle_names}")
        print(f"[Exploration] Found locations: {location_names}")

        # Get robot's starting location if possible
        robot_pos = None
        try:
            # Try to get the robot's starting position from the obs or 'robot' key
            robot_keys = [k for k in positions if 'robot' in k]
            if robot_keys:
                robot_pos = positions[robot_keys[0]]
            else:
                # Fallback: check location from 'obs', e.g. obs['robot_pos'] or similar
                robot_pos = obs.get('robot_pos', None)
        except Exception as e:
            robot_pos = None

        # Exploration strategy:
        # Try moving to each location, "identifying" objects, pulling drawers to see lock state, etc.
        # For each type of relevant available skill, attempt a call where parameters are plausible

        # Store results for further analysis
        exploration_outcomes = {}

        # Attempt execute_go (move the robot to different locations)
        for loc in location_names:
            try:
                print(f"[Exploration] Trying execute_go to location: {loc}")
                obs, reward, done = execute_go(env, task, from_loc=robot_pos, to_loc=loc)
                # Update robot position for later steps
                robot_pos = loc
                exploration_outcomes[f"execute_go:{loc}"] = (obs, reward, done)
            except Exception as e:
                print(f"[Exploration] execute_go to {loc} failed: {e}")

        # Try picking up objects (if hand is empty and object is on the floor)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_pick of {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    exploration_outcomes[f"execute_pick:{obj}@{loc}"] = (obs, reward, done)
                except Exception as e:
                    print(f"[Exploration] execute_pick {obj} at {loc} failed: {e}")

        # Try pulling drawer using handle if "holding handle" is possible
        for drawer in drawer_names:
            for handle in handle_names:
                for loc in location_names:
                    try:
                        # Try to pick the handle first
                        try:
                            print(f"[Exploration] Trying execute_pick of handle {handle} at {loc}")
                            obs, reward, done = execute_pick(env, task, handle, loc)
                        except Exception as ep:
                            print(f"[Exploration] execute_pick of handle {handle} failed: {ep}")
                        # Try to pull after picking handle
                        print(f"[Exploration] Trying execute_pull on drawer {drawer} with handle {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        exploration_outcomes[f"execute_pull:{drawer}:{handle}@{loc}"] = (obs, reward, done)
                    except Exception as e:
                        print(f"[Exploration] execute_pull {drawer} with handle {handle} at {loc} failed: {e}")

        # Try pushing drawers (close if open)
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_push drawer {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    exploration_outcomes[f"execute_push:{drawer}@{loc}"] = (obs, reward, done)
                except Exception as e:
                    print(f"[Exploration] execute_push {drawer} at {loc} failed: {e}")

        # Try placing objects in drawers if possible
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_place {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        exploration_outcomes[f"execute_place:{obj}:{drawer}@{loc}"] = (obs, reward, done)
                    except Exception as e:
                        print(f"[Exploration] execute_place {obj} in {drawer} at {loc} failed: {e}")

        # Try sweeping all objects at all locations (for completeness)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_sweep object {obj} at location {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    exploration_outcomes[f"execute_sweep:{obj}@{loc}"] = (obs, reward, done)
                except Exception as e:
                    print(f"[Exploration] execute_sweep {obj} at {loc} failed: {e}")

        # Try primitive manipulation (gripper action)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            exploration_outcomes["execute_gripper"] = (obs, reward, done)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Outcomes summary:")
        for act, result in exploration_outcomes.items():
            print(f"  Action {act} -> Reward: {result[1]}, Done: {result[2]}")

        print("[Exploration] Exploration phase complete. Analyze these results to infer missing predicates (e.g., 'lock-known', 'identified', etc.) as per feedback.")

        # =========================
        # --- End of Exploration --
        # =========================

        # At this point, based on outcomes above, one can infer (offline, via logs) which
        # predicate(s) are not present or are missing for goal achievement, and revise domain if needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
