# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use ONLY pre-existing skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Returns: dictionary mapping object names to their positions and optionally their properties
        positions = get_object_positions()
        
        # ======== PREDICATE EXPLORATION PHASE =======
        # Because plan validation timed out according to feedback (likely due to a missing precondition or ungrounded predicate),
        # we must run an EXPLORATION phase to identify which facts/predicates are missing for our planner/domain.
        #
        # We will attempt to call each available skill with safe arguments,
        # observe any errors, and then (optionally) log results.
        # The aim is to identify if we are missing required facts (e.g., not all objects/locations are known).
        #
        # We'll do this in a data-driven way, using all available skill names from the provided list.

        # Map available skill names to their expected arguments (based on PDDL domain)
        # Only skills from skill_code are to be used/called
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Get candidate objects, drawers, handles, locations from positions
        # In a generic domain, we infer types from naming or position dictionaries
        # keys: {'object1': pos, 'drawer1': pos, 'handle1': pos, ...}
        object_names = [k for k in positions if 'object' in k]
        drawer_names = [k for k in positions if 'drawer' in k]
        handle_names = [k for k in positions if 'handle' in k]
        location_names = [k for k in positions if 'location' in k or 'room' in k or 'desk' in k]
        # As fallback, include any name not categorized as other type
        other_names = list(set(positions.keys()) - set(object_names) - set(drawer_names) - set(handle_names))

        # Heuristic: likely locations are also target positions for robot
        # If locations list is empty, add any 'desk', 'table', or 'room' keys
        if not location_names:
            for k in positions.keys():
                if 'desk' in k or 'table' in k or 'room' in k:
                    location_names.append(k)
        # Fallback: if still empty, use any key as location (for completeness)
        if not location_names:
            location_names = list(positions.keys())

        # Pick the first available names for sample exploration
        sample_obj = object_names[0] if object_names else None
        sample_drawer = drawer_names[0] if drawer_names else None
        sample_handle = handle_names[0] if handle_names else None
        sample_location = location_names[0] if location_names else None
        sample_second_location = location_names[1] if len(location_names) > 1 else None

        # Store explored predicates and errors
        exploration_results = []
        print("===== EXPLORATION PHASE: Calling candidate skills to identify missing predicates =====")
        for skill in available_skills:
            try:
                if skill == 'execute_pick':
                    if sample_obj and sample_location:
                        print(f"[Exploration] Trying {skill}({sample_obj}, {sample_location})...")
                        # Try picking up the object at its location
                        obs, reward, done = execute_pick(env, task, target_name=sample_obj, target_pos=positions[sample_obj])
                        exploration_results.append((skill, 'success'))
                elif skill == 'execute_place':
                    if sample_obj and sample_drawer and sample_location:
                        print(f"[Exploration] Trying {skill}({sample_obj}, {sample_drawer}, {sample_location})...")
                        obs, reward, done = execute_place(env, task, target_name=sample_obj, target_drawer=sample_drawer, target_pos=positions[sample_drawer])
                        exploration_results.append((skill, 'success'))
                elif skill == 'execute_push':
                    if sample_drawer and sample_location:
                        print(f"[Exploration] Trying {skill}({sample_drawer}, {sample_location})...")
                        obs, reward, done = execute_push(env, task, target_drawer=sample_drawer, target_pos=positions[sample_drawer])
                        exploration_results.append((skill, 'success'))
                elif skill == 'execute_pull':
                    # Needs a handle and location, and often the handle is a gripped object
                    if sample_drawer and sample_handle and sample_location:
                        print(f"[Exploration] Trying {skill}({sample_drawer}, {sample_handle}, {sample_location})...")
                        obs, reward, done = execute_pull(env, task, target_drawer=sample_drawer, target_handle=sample_handle, target_pos=positions[sample_drawer])
                        exploration_results.append((skill, 'success'))
                elif skill == 'execute_sweep':
                    if sample_obj and sample_location:
                        print(f"[Exploration] Trying {skill}({sample_obj}, {sample_location})...")
                        obs, reward, done = execute_sweep(env, task, target_name=sample_obj, target_pos=positions[sample_obj])
                        exploration_results.append((skill, 'success'))
                elif skill == 'execute_rotate':
                    # Assume this is a gripper/robot base rotation, needs only env/task
                    print(f"[Exploration] Trying {skill}()...")
                    obs, reward, done = execute_rotate(env, task)
                    exploration_results.append((skill, 'success'))
                elif skill == 'execute_go':
                    if sample_location and sample_second_location:
                        print(f"[Exploration] Trying {skill}({sample_location}, {sample_second_location})...")
                        obs, reward, done = execute_go(env, task, from_location=sample_location, to_location=sample_second_location)
                        exploration_results.append((skill, 'success'))
                elif skill == 'execute_gripper':
                    print(f"[Exploration] Trying {skill}()...")
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results.append((skill, 'success'))
            except Exception as e:
                print(f"[Exploration][{skill}] Exception occurred: {e}")
                exploration_results.append((skill, f'error: {str(e)}'))

        # At the end, print out the summary
        print("\n====== Exploration Summary ======")
        for skill, result in exploration_results:
            print(f"{skill}: {result}")
        print("\n--- Exploration complete. Review results above for missing predicates or problematic steps. ---\n")

        # After identifying missing predicates or incomplete facts, the user should adjust their initial state, domain, or plan accordingly.
        # At this stage we STOP -- the remainder (execution of intended plan/oracle) should start only when all predicates and initial state issues are resolved.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
