# run_skeleton_task.py (Executable Code with Exploration Phase)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing skills, do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with an exploration phase to detect missing predicates as required.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for video recording (if required)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()  # {'obj': (x, y, z), ...}

        # =============== EXPLORATION PHASE ===============
        # Feedback and setup suggest there might be a missing predicate (e.g. "lock-known" for a drawer)
        # This phase is to explicitly call existing skills to gather information and "unlock"/learn the state if required.

        print("[Exploration] Entering exploration phase to discover missing predicates (e.g., lock-known about handles/drawers).")
        exploration_success = False
        explored_objs = []
        explored_drawers = []
        explored_handles = []

        # Attempt to use available skills to explore/identify lock state.
        # We'll try execute_pull (as in exploration-PDDL it is used to discover 'lock-known' and it's available in the skill list).

        # Try to explore each handle (assume keys in object positions that are likely handles and drawers)
        # Example naming conventions: 'handle_x', 'drawer_x', etc.

        for name in positions.keys():
            if "handle" in name and name not in explored_handles:
                handle_name = name
                handle_pos = positions[handle_name]
                # Try to find which drawer this handle might belong to
                corresponding_drawer = None
                for k in positions.keys():
                    if "drawer" in k and k not in explored_drawers:
                        corresponding_drawer = k
                        drawer_pos = positions[corresponding_drawer]
                        break
                if corresponding_drawer is None:
                    continue
                # Try to go to handle position and pick it (execute_pick), then pull (execute_pull)
                print(f"[Exploration] Attempting to pick and pull handle '{handle_name}' associated to drawer '{corresponding_drawer}'.")

                try:
                    # 1. Move robot to handle location (execute_go)
                    obs, reward, done = execute_go(env, task, target_pos=handle_pos)
                    if done:
                        print(f"[Exploration] Robot reached handle {handle_name} at position {handle_pos}")

                    # 2. Pick up handle (execute_pick)
                    obs, reward, done = execute_pick(env, task, handle_name, handle_pos)
                    if done:
                        print(f"[Exploration] Picked up handle '{handle_name}'.")

                    # 3. Attempt to pull (execute_pull): requires drawer name, handle name, and location
                    obs, reward, done = execute_pull(env, task, corresponding_drawer, handle_name, handle_pos)
                    if done:
                        print(f"[Exploration] Pulled on handle '{handle_name}'. Lock state for drawer '{corresponding_drawer}' may now be known.")
                        exploration_success = True
                        explored_handles.append(handle_name)
                        explored_drawers.append(corresponding_drawer)
                        break
                except Exception as e:
                    print(f"[Exploration] Exception during exploration with handle '{handle_name}': {e}")
        if exploration_success:
            print("[Exploration] Exploration phase completed: successfully attempted to discover missing predicates (e.g., lock-known).")
        else:
            print("[Exploration] Exploration failed: could not successfully discover missing predicates (some drawer/handle may be unavailable or skill failed).")


        # =============== END OF EXPLORATION PHASE =============

        # =============== MAIN TASK PHASE (example) ===============
        # Now the robot should be able to proceed, as missing predicates should be "known" after exploration.

        # Placeholder for further logical steps of your task.
        # You'll need to fill in the details according to your specific task and available objects.
        # e.g., pick up an object, move it to a new location, open drawer, place object etc.

        # EXAMPLE CONTROL FLOW (to be customized for your scenario):

        # For each object, if it's on the floor and within reach, pick and place in the first available open drawer.
        for obj_name, obj_pos in positions.items():
            if "object" in obj_name or "obj" in obj_name:
                # Try to pick up the object (execute_pick)
                try:
                    print(f"[Task] Attempting to pick object '{obj_name}' at {obj_pos}.")
                    obs, reward, done = execute_pick(env, task, obj_name, obj_pos)
                    if done:
                        print(f"[Task] Successfully picked object '{obj_name}'.")
                    else:
                        continue
                except Exception as e:
                    print(f"[Task] Failed picking object '{obj_name}': {e}")
                    continue

                # Now find a drawer that is open and not full for placing
                target_drawer = None
                target_drawer_pos = None
                for drawer_name, pos in positions.items():
                    if "drawer" in drawer_name and drawer_name not in explored_drawers:
                        target_drawer = drawer_name
                        target_drawer_pos = pos
                        break

                if not target_drawer:
                    print(f"[Task] No suitable drawer found for placing '{obj_name}'.")
                    continue

                try:
                    # Move robot to drawer (if not there already)
                    obs, reward, done = execute_go(env, task, target_pos=target_drawer_pos)
                    # Place the object into the drawer (execute_place)
                    obs, reward, done = execute_place(env, task, obj_name, target_drawer, target_drawer_pos)
                    if done:
                        print(f"[Task] Placed object '{obj_name}' into drawer '{target_drawer}'.")
                    else:
                        print(f"[Task] Failed to place object '{obj_name}' into '{target_drawer}'.")
                except Exception as e:
                    print(f"[Task] Exception while placing '{obj_name}': {e}")
        # END EXAMPLE

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
