# run_skeleton_task.py (Completed with Exploration Phase for Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ----- EXPLORATION PHASE: Discovering Missing Predicate -----
        # We need to determine, by action, if a required predicate is missing for plan execution.
        # Approach: Try all available skills/actions in plausible configurations to see if any action fails to execute.
        # If a skill fails or does nothing, this hints at a missing predicate in init or the domain.
        skill_names = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Gather information about what objects, drawers, handles, and locations exist
        # These keys may need to be adapted to your environment naming conventions
        objects = [k for k, v in positions.items() if 'object' in k and 'handle' not in k]
        drawers = [k for k, v in positions.items() if 'drawer' in k]
        handles = [k for k, v in positions.items() if 'handle' in k]
        locations = [k for k, v in positions.items() if 'location' in k or 'table' in k or 'room' in k]
        # Fallback: just use all unique location values if location names not found
        if not locations:
            locations = list(set([tuple(pos) for pos in positions.values()]))

        # Helper: Get robot's current location if available
        robot_loc = None
        try:
            robot_loc = [k for k in positions if 'robot' in k and 'location' in k]
            if robot_loc:
                robot_loc = robot_loc[0]
            else:
                # As fallback, use 'base' or any location
                robot_loc = locations[0] if locations else None
        except Exception:
            robot_loc = None
        
        print("=== Exploration Phase: Executing All Available Skills to Detect Missing Predicate ===")
        exploration_success = {}
        for skill in skill_names:
            try:
                print(f"[Exploration] Trying skill: {skill}")
                if skill == 'execute_pick':
                    # Try picking all objects from all plausible locations
                    for o in objects:
                        # Assume robot must be near object; try all locations if not sure
                        for loc in locations:
                            try:
                                obs, reward, done = execute_pick(env, task, target_obj=o, target_loc=loc)
                                print(f"  execute_pick({o}, {loc}) => Success")
                                exploration_success[('execute_pick', o, loc)] = True
                            except Exception as e:
                                print(f"  execute_pick({o}, {loc}) failed: {e}")
                                exploration_success[('execute_pick', o, loc)] = False
                elif skill == 'execute_place':
                    # Try placing each object into each drawer from all locations
                    for o in objects:
                        for d in drawers:
                            for loc in locations:
                                try:
                                    obs, reward, done = execute_place(env, task, target_obj=o, target_drawer=d, target_loc=loc)
                                    print(f"  execute_place({o}, {d}, {loc}) => Success")
                                    exploration_success[('execute_place', o, d, loc)] = True
                                except Exception as e:
                                    print(f"  execute_place({o}, {d}, {loc}) failed: {e}")
                                    exploration_success[('execute_place', o, d, loc)] = False
                elif skill == 'execute_push':
                    # Try pushing each drawer from all locations
                    for d in drawers:
                        for loc in locations:
                            try:
                                obs, reward, done = execute_push(env, task, target_drawer=d, target_loc=loc)
                                print(f"  execute_push({d}, {loc}) => Success")
                                exploration_success[('execute_push', d, loc)] = True
                            except Exception as e:
                                print(f"  execute_push({d}, {loc}) failed: {e}")
                                exploration_success[('execute_push', d, loc)] = False
                elif skill == 'execute_pull':
                    # Try pulling each drawer handle from all locations (need to pick handle first if required)
                    for h in handles:
                        for d in drawers:
                            for loc in locations:
                                try:
                                    obs, reward, done = execute_pull(env, task, target_drawer=d, target_handle=h, target_loc=loc)
                                    print(f"  execute_pull({d}, {h}, {loc}) => Success")
                                    exploration_success[('execute_pull', d, h, loc)] = True
                                except Exception as e:
                                    print(f"  execute_pull({d}, {h}, {loc}) failed: {e}")
                                    exploration_success[('execute_pull', d, h, loc)] = False
                elif skill == 'execute_sweep':
                    # Try sweeping all objects from all locations
                    for o in objects:
                        for loc in locations:
                            try:
                                obs, reward, done = execute_sweep(env, task, target_obj=o, target_loc=loc)
                                print(f"  execute_sweep({o}, {loc}) => Success")
                                exploration_success[('execute_sweep', o, loc)] = True
                            except Exception as e:
                                print(f"  execute_sweep({o}, {loc}) failed: {e}")
                                exploration_success[('execute_sweep', o, loc)] = False
                elif skill == 'execute_rotate':
                    # If your domain supports rotating, attempt rotation for all handles or drawers
                    for h in handles:
                        try:
                            obs, reward, done = execute_rotate(env, task, target_handle=h)
                            print(f"  execute_rotate({h}) => Success")
                            exploration_success[('execute_rotate', h)] = True
                        except Exception as e:
                            print(f"  execute_rotate({h}) failed: {e}")
                            exploration_success[('execute_rotate', h)] = False
                elif skill == 'execute_go':
                    # Move between all pairs of locations
                    for from_loc in locations:
                        for to_loc in locations:
                            if from_loc == to_loc:
                                continue
                            try:
                                obs, reward, done = execute_go(env, task, from_loc=from_loc, to_loc=to_loc)
                                print(f"  execute_go({from_loc}, {to_loc}) => Success")
                                exploration_success[('execute_go', from_loc, to_loc)] = True
                            except Exception as e:
                                print(f"  execute_go({from_loc}, {to_loc}) failed: {e}")
                                exploration_success[('execute_go', from_loc, to_loc)] = False
                elif skill == 'execute_gripper':
                    # Use the gripper if available
                    try:
                        obs, reward, done = execute_gripper(env, task)
                        print(f"  execute_gripper() => Success")
                        exploration_success[('execute_gripper',)] = True
                    except Exception as e:
                        print(f"  execute_gripper() failed: {e}")
                        exploration_success[('execute_gripper',)] = False
            except Exception as exc:
                print(f"Exception during skill exploration {skill}: {exc}")

        # === SUMMARY OF EXPLORATION ===
        print("=== Exploration Completed ===")
        for key, val in exploration_success.items():
            if not val:
                print(f"  [Missing Predicate?] Action: {key} failed.")
        print("Use above output to infer any missing predicates needed for planning.")

        # === TASK-ORACLE PLAN EXECUTION (Placeholder) ===
        # Here, you would insert the actual step-by-step logic for the oracle plan
        # using the skills defined above, matching your task specification.
        # Example:
        #
        # obs, reward, done = execute_go(env, task, from_loc='A', to_loc='B')
        # obs, reward, done = execute_pick(env, task, target_obj='object1', target_loc='B')
        # obs, reward, done = execute_place(env, task, target_obj='object1', target_drawer='drawer1', target_loc='B')
        #
        # Make sure to check for exceptions and whether each step completes as expected.

        # [Task End Condition]
        # If your task has a goal/done state, use it here and break/return as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
