# run_skeleton_task.py (Completed for predicate exploration and action execution)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined primitives; do not redefine
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Since verbose exploration is needed for missing predicates,
        # we'll perform an exploration phase to help the user identify
        # which predicate or relation(s) are missing in the problem/plan.

        # 1) List available skills (from skill_code and domain PDDL)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]
        
        print("[Exploration] Starting predicate exploration.")

        # 2) Try critical skills with known preconditions. If an exception
        # is raised, it may be due to a missing predicate or domain issue.
        # We'll attempt to use each skill in sequence on example objects.

        # For demonstration, we'll attempt to:
        #   - Move (execute_go)
        #   - Pick (execute_pick)
        #   - Open/Close (execute_pull/execute_push)
        #   - Place (execute_place)

        # -- Identify useful objects, drawers, locations & handles from positions --
        # positions: Dict[str, (x,y,z)], e.g., {'drawer1': (..), 'apple1': (..), ...}

        # Dummy selectors - replace with your simulation's actual object
        # names or use some heuristics to detect types from naming.
        objects = [name for name in positions if 'drawer' not in name and 'handle' not in name]
        drawers = [name for name in positions if 'drawer' in name]
        handles = [name for name in positions if 'handle' in name]
        locations = list(positions.keys())  # All named items as possible locations

        # Pick first found, fallback handling if not found
        test_object = objects[0] if objects else None
        test_drawer = drawers[0] if drawers else None
        test_location = locations[0] if locations else None
        test_handle = handles[0] if handles else None

        if not all([test_object, test_drawer, test_location]):
            print("[Error] Not all required objects (object, drawer, location) found in environment. Exploration skipped.")
            return

        # Determining the robot's initial location
        robot_location = None
        for loc in locations:
            if 'robot' in loc or 'start' in loc or 'init' in loc:
                robot_location = loc
                break
        if not robot_location:
            # Fallback: just use test_location
            robot_location = test_location

        # === Run exploration actions ===
        exploration_success = {}
        for skill in available_skills:
            try:
                print("[Exploration] Attempting skill:", skill)
                if skill == 'execute_go':
                    # Move robot from robot_location to test_location (if distinct)
                    target_location = test_location if robot_location != test_location else locations[-1]
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=robot_location,
                        to_location=target_location
                    )
                elif skill == 'execute_pick':
                    # Pick up the test_object at test_location
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=test_object,
                        location_name=test_location
                    )
                elif skill == 'execute_pull':
                    # Attempt to pull open the drawer using the handle
                    if test_handle:
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            drawer_name=test_drawer,
                            handle_name=test_handle,
                            location_name=test_location
                        )
                    else:
                        print("[Exploration] No handle found for pull; skipping.")
                        continue
                elif skill == 'execute_push':
                    # Try pushing the drawer closed at test_location
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name=test_drawer,
                        location_name=test_location
                    )
                elif skill == 'execute_place':
                    # Place object in drawer at test_location
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj_name=test_object,
                        drawer_name=test_drawer,
                        location_name=test_location
                    )
                elif skill == 'execute_sweep':
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        obj_name=test_object,
                        location_name=test_location
                    )
                elif skill == 'execute_gripper':
                    obs, reward, done = execute_gripper(env, task)
                elif skill == 'execute_rotate':
                    # Not implemented in domain; skipping
                    print("[Exploration] execute_rotate not implemented in domain, skipping.")
                    continue
                exploration_success[skill] = True
                print("[Exploration] Skill", skill, "succeeded.")
            except Exception as e:
                exploration_success[skill] = False
                print("[Exploration] Skill", skill, "failed with exception:", str(e))

        # Summarize results for missing predicate detection
        print("[Exploration] Summary of skill execution results:")
        for skill, success in exploration_success.items():
            print("   -", skill, ":", "SUCCESS" if success else "FAILED")

        print("[Exploration] If a skill failed with a predicate-related exception,")
        print("please check that all predicates required by the domain for that action")
        print("are available in the problem definition.")

        # You can now use this analysis to check which action's preconditions are unmet,
        # and which predicate is thus missing in your problem or plan specification.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
