# run_skeleton_task.py (Filled)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use available skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # The following is an exploration phase required by feedback: detecting missing predicate (drawer-open drawer1)
        # We'll do it by attempting to open or interact with drawer1 and observe if feedback occurs

        print("[Exploration] Start exploring missing predicate related to drawer1.")

        # The available skills from skill_code
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this generic domain, typical exploration:
        # 1. Go to the location of the drawer.
        # 2. Try to open the drawer with execute_pull.
        # 3. If feedback says the drawer is already open (predicate present), note this.
        # 4. Otherwise, attempt to open; if fails, print exception.
        # We assume the presence of 'drawer1' and its 'handle' in the environment.

        # Step 1: Get the location for drawer1 and its handle object
        try:
            drawer_name = 'drawer1'
            handle_name = 'drawer1_handle'
            # Name conventions may vary, handle naming accordingly
            drawer_loc = positions.get(drawer_name)
            handle_pos = positions.get(handle_name)
            robot_pos = None
            # Determine robot's current location from observation if possible
            for k in positions:
                if 'robot' in k:
                    robot_pos = positions[k]
                    break
            if robot_pos is None:
                # Fallback: guess 'home' or first location
                robot_pos = list(positions.values())[0]

            # Step 2: Move to drawer location if not already there
            if robot_pos != drawer_loc:
                print(f"[Exploration] Moving robot from {robot_pos} to drawer {drawer_name} at {drawer_loc}.")
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=robot_pos,
                        to_location=drawer_loc,
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Exploration] Failed to move: {e}")

            # Step 3: Try to pick the handle
            print(f"[Exploration] Picking handle '{handle_name}'.")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=handle_name,
                    location=drawer_loc,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to pick handle: {e}")

            # Step 4: Try to pull (open) the drawer with the handle
            print(f"[Exploration] Attempting to open (pull) drawer '{drawer_name}' via handle '{handle_name}'.")
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_name=handle_name,
                    location=drawer_loc,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to pull drawer: {e}")
                print("[Exploration] This might reveal if the drawer is already open or missing 'drawer-open' predicate.")

            print("[Exploration] Finished. Check environment and feedback for the predicate (drawer-open drawer1).")

        except Exception as e:
            print(f"[Exploration] Exploration error: {e}")

        # === End of exploration phase ===
        #
        # If more actions needed to solve the overall task, e.g. placing an object into drawer, they can be added here
        # whenever precondition for 'drawer-open' is met.

        # Example: Try placing an object in the drawer if it's open
        try:
            some_object = None
            # pick an object on the floor from init state if available
            for obj in positions:
                if obj.startswith('obj') or obj.startswith('object'):
                    some_object = obj
                    break
            if some_object is not None and drawer_loc is not None:
                # Try pick object
                print(f"[Task] Try picking up object {some_object} at {drawer_loc}.")
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=some_object,
                        location=drawer_loc,
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Task] Could not pick up {some_object}: {e}")
                # Try place in drawer
                print(f"[Task] Try placing object {some_object} in drawer {drawer_name}.")
                try:
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=some_object,
                        drawer_name=drawer_name,
                        location=drawer_loc,
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Task] Could not place {some_object}: {e}")
        except Exception as e:
            print(f"[Task] Error in pick-place phase: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()