# run_skeleton_task.py (Completed for Missing Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all predefined skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Dictionary example: {'drawer1': (x, y, z), 'handle1': (x, y, z), ...}
        positions = get_object_positions()
        
        # ---------------------- Exploration Phase ----------------------
        # Purpose: Detect missing (drawer-closed ?d) predicate via physical/skill probing

        # Key idea: The feedback "(drawer-closed drawer1)" indicates that the 'drawer-closed' predicate may not be observable/perceived directly,
        # so we attempt to perform actions on 'drawer1' that require its state, using predefined skills

        # The process: 
        # 1. Try to open (pull) 'drawer1' (which may only be possible if it's closed & unlocked).
        # 2. If this fails, something about the locking/closed state is missing; try other actions and log feedback.

        drawer_name = 'drawer1'
        handle_name = 'handle1'
        try:
            drawer_pos = positions[drawer_name]
            handle_pos = positions[handle_name]
        except KeyError:
            print(f"[Error] Could not find position for '{drawer_name}' or '{handle_name}'. Aborting exploration.")
            return

        robot_loc = None
        # Try to determine current robot position/location
        # We'll try to infer a key from positions (e.g., 'robot', 'base', etc.)
        for key in positions:
            if 'robot' in key:
                robot_loc = key
                break
        if robot_loc is None:
            # Try fallback
            robot_loc = list(positions.keys())[0]  # assume first key is a location

        print(f"[Exploration] Assuming robot is at '{robot_loc}'.")
        
        # PHASE 1: Approach drawer handle (if necessary)
        # Use execute_go to go from robot_loc to handle/drawer location if not already there
        try:
            # Always go to handle location for exploration - sometimes the planner needs this for preconditions
            obs, reward, done = execute_go(env, task, robot_loc, handle_name)
            print(f"[Exploration] Moved robot from '{robot_loc}' to '{handle_name}' for handle access.")
            robot_loc = handle_name
        except Exception as e:
            print(f"[Exploration] execute_go failed (robot may already be at handle): {e}")

        # PHASE 2: Pick the handle (to prepare for pull)
        try:
            obs, reward, done = execute_pick(env, task, handle_name, robot_loc)
            print(f"[Exploration] Picked up '{handle_name}'.")
        except Exception as e:
            print(f"[Exploration] Failed to pick handle '{handle_name}': {e}")

        # PHASE 3: Attempt to pull the drawer using the handle
        missing_predicate_found = False
        try:
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_loc)
            print(f"[Exploration] Successfully pulled '{drawer_name}'. Drawer might not have been locked and was closed.")
        except Exception as e:
            print(f"[Exploration] Could NOT pull '{drawer_name}' at location '{robot_loc}': {e}")
            print("[Exploration] This may indicate a missing predicate such as (drawer-closed drawer1) or (drawer-locked drawer1) in the environment state.")
            missing_predicate_found = True

        # PHASE 4: If failed, try additional skills to probe
        if missing_predicate_found:
            # Try to sweep or gripper as additional exploration; may reveal other preconditions
            try:
                obs, reward, done = execute_sweep(env, task, drawer_name, robot_loc)
                print(f"[Exploration] Executed sweep on '{drawer_name}'.")
            except Exception as e2:
                print(f"[Exploration] Sweep on '{drawer_name}' failed: {e2}")
            try:
                obs, reward, done = execute_gripper(env, task)
                print(f"[Exploration] Executed gripper skill.")
            except Exception as e3:
                print(f"[Exploration] Gripper skill failed: {e3}")

        # (Optional) After the exploration phase, your logic could branch into the main task using the information gathered.
        # For this skeleton, we stop after exploration/probing for missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()