# run_skeleton_task.py (Fully Generic Skeleton with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills, do not redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration to resolve missing predicates.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Debug] Object positions:", positions)

        # === EXPLORATION PHASE - Attempt to discover missing predicates or state ===
        print("[Exploration] Starting exploration phase to resolve missing predicates.")

        # The following logic tries to check which predicate might be missing in the state.
        # Since no explicit oracle plan is provided or problem instance, we treat this phase generically:
        # 1. For each object in the environment, try the available skills and catch failures indicative of missing predicates!

        # Identify available objects, drawers, locations from the positions dictionary or descriptions
        # These will depend on your environment instance
        # For demonstration, collect lists of objects by type based on 'positions' keys
        all_objects = []
        all_drawers = []
        all_locations = set()
        handle_objects = []
        for name in positions:
            if "drawer" in name:
                all_drawers.append(name)
            elif "handle" in name:
                handle_objects.append(name)
            else:
                all_objects.append(name)
            # Try to deduce possible location keys by splitting "_at_" or "_pos"
            if isinstance(positions[name], dict):
                for k in positions[name]:
                    all_locations.add(k)
            else:
                all_locations.add("default_location") # fallback
                
        all_locations = list(all_locations)
        if not all_locations:
            all_locations = ["default_location"]
        print(f"[Exploration] Found objects: {all_objects}")
        print(f"[Exploration] Found drawers: {all_drawers}")
        print(f"[Exploration] Found handles: {handle_objects}")
        print(f"[Exploration] Found locations: {all_locations}")

        # Simulate typical sequence for discovering missing predicate(s)
        exploration_successful = False
        for obj in all_objects:
            for loc in all_locations:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc}")
                    result = execute_pick(env, task, obj, loc)
                    print(f"  [Exploration] Picked {obj} at {loc}: {result}")
                    exploration_successful = True
                    # Now try placing in each drawer if possible
                    for drawer in all_drawers:
                        print(f"[Exploration] Trying to place {obj} in drawer {drawer} at {loc}")
                        try:
                            result2 = execute_place(env, task, obj, drawer, loc)
                            print(f"  [Exploration] Placed {obj} in {drawer}: {result2}")
                            exploration_successful = True
                        except Exception as e2:
                            print(f"  [Exploration] Failed to place {obj} in {drawer}: {str(e2)}")
                except Exception as e:
                    print(f"  [Exploration] Failed to pick {obj} at {loc}: {str(e)}")

        # Try opening drawers by pulling handles (requires holding handle object)
        for drawer in all_drawers:
            for handle in handle_objects:
                for loc in all_locations:
                    try:
                        print(f"[Exploration] Trying to pick handle {handle} at {loc}")
                        execute_pick(env, task, handle, loc)
                        print(f"[Exploration] Picked handle {handle} at {loc}")
                        print(f"[Exploration] Trying to pull drawer {drawer} by handle {handle} at {loc}")
                        execute_pull(env, task, drawer, handle, loc)
                        print(f"  [Exploration] Pulled open drawer {drawer} by {handle} at {loc}")
                        exploration_successful = True
                    except Exception as e:
                        print(f"  [Exploration] Failed to pull {drawer} with {handle} at {loc}: {str(e)}")

        # Try moving between locations
        if len(all_locations) > 1:
            for from_loc in all_locations:
                for to_loc in all_locations:
                    if from_loc == to_loc:
                        continue
                    try:
                        print(f"[Exploration] Trying to move from {from_loc} to {to_loc}")
                        execute_go(env, task, from_loc, to_loc)
                        print(f"  [Exploration] Moved from {from_loc} to {to_loc}")
                        exploration_successful = True
                    except Exception as e:
                        print(f"  [Exploration] Failed to move from {from_loc} to {to_loc}: {str(e)}")

        # Try pushing (closing) each drawer
        for drawer in all_drawers:
            for loc in all_locations:
                try:
                    print(f"[Exploration] Trying to push (close) drawer {drawer} at {loc}")
                    execute_push(env, task, drawer, loc)
                    print(f"  [Exploration] Pushed (closed) drawer {drawer} at {loc}")
                    exploration_successful = True
                except Exception as e:
                    print(f"  [Exploration] Failed to push {drawer} at {loc}: {str(e)}")

        # Try sweeping objects
        for obj in all_objects:
            for loc in all_locations:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc}")
                    execute_sweep(env, task, obj, loc)
                    print(f"  [Exploration] Swept {obj} at {loc}")
                    exploration_successful = True
                except Exception as e:
                    print(f"  [Exploration] Failed to sweep {obj} at {loc}: {str(e)}")

        # Try execute_gripper if available
        try:
            print(f"[Exploration] Trying to execute_gripper")
            execute_gripper(env, task)
            print(f"  [Exploration] execute_gripper success")
            exploration_successful = True
        except Exception as e:
            print(f"  [Exploration] Failed execute_gripper: {str(e)}")

        if not exploration_successful:
            print("[Exploration] No exploration actions were successful. There may be fundamental missing predicates or failed skill mappings.")

        print("[Exploration] Finished. Review above trace to localize what predicate(s) might be missing.")

        # === TASK PHASE ===
        # After exploration, in your actual experiment, you would now execute the oracle plan using only the validated skills.
        # This code is a template – plug the actual task steps as a sequence of skill calls as in the exploration above.

        # For example, an oracle plan (pseudocode) could look like:
        # obs = execute_go(env, task, 'locA', 'locB')
        # obs = execute_pick(env, task, 'ball_1', 'locB')
        # obs = execute_place(env, task, 'ball_1', 'drawer_1', 'locB')
        # ...etc...

        print("[Task] (Placeholder) Now execute your verified oracle plan here using available skills.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
