# run_skeleton_task.py (Executable Code Completing the Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined/existing skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'drawer1': (x, y, z), 'obj1': (x, y, z), ...}

        # === Begin Exploration to Find Missing Predicate or System Problem ===
        # The problem feedback indicates there is likely a missing predicate/condition in the environment or plan.
        # To address this, we run the available skills in sequence as an exploration routine to check for failures or unmodeled preconditions.
        # We do NOT invent new skills or predicates here; we simply execute what is available.

        # Make a list of available objects, drawers, locations if present in object_positions
        drawer_names = [name for name in positions if 'drawer' in name]
        object_names = [name for name in positions if 'obj' in name or 'object' in name]
        location_names = [name for name in positions if 'loc' in name or 'room' in name or 'area' in name]
        handle_names = [name for name in positions if 'handle' in name]

        # Fallback if the above lists are empty (for robustness): use all keys
        all_names = list(positions.keys())

        # Attempt each skill in a minimal way to probe the environment
        print("===== [Exploration Phase] Executing Available Skills... =====")
        skill_results = {}
        # Try execute_pick on first available object
        pick_obj = object_names[0] if object_names else (all_names[0] if all_names else None)
        pick_loc = location_names[0] if location_names else (all_names[0] if all_names else None)
        if pick_obj and pick_loc:
            print(f"Trying execute_pick with {pick_obj} at {pick_loc}")
            try:
                obs, reward, done = execute_pick(env, task, pick_obj, pick_loc)
                skill_results['execute_pick'] = (obs, reward, done)
                print("[SUCCESS] execute_pick")
            except Exception as e:
                skill_results['execute_pick'] = str(e)
                print(f"[FAIL] execute_pick: {e}")

        # Try execute_place if possible
        if drawer_names and pick_obj and pick_loc:
            drawer = drawer_names[0]
            try:
                print(f"Trying execute_place with {pick_obj} into {drawer} at {pick_loc}")
                obs, reward, done = execute_place(env, task, pick_obj, drawer, pick_loc)
                skill_results['execute_place'] = (obs, reward, done)
                print("[SUCCESS] execute_place")
            except Exception as e:
                skill_results['execute_place'] = str(e)
                print(f"[FAIL] execute_place: {e}")

        # Try execute_push on drawer
        if drawer_names and pick_loc:
            drawer = drawer_names[0]
            try:
                print(f"Trying execute_push on {drawer} at {pick_loc}")
                obs, reward, done = execute_push(env, task, drawer, pick_loc)
                skill_results['execute_push'] = (obs, reward, done)
                print("[SUCCESS] execute_push")
            except Exception as e:
                skill_results['execute_push'] = str(e)
                print(f"[FAIL] execute_push: {e}")

        # Try execute_pull, using a handle if available, else object
        pull_drawer = drawer_names[0] if drawer_names else None
        pull_handle = handle_names[0] if handle_names else (object_names[0] if object_names else None)
        if pull_drawer and pull_handle and pick_loc:
            try:
                print(f"Trying execute_pull on {pull_drawer} with {pull_handle} at {pick_loc}")
                obs, reward, done = execute_pull(env, task, pull_drawer, pull_handle, pick_loc)
                skill_results['execute_pull'] = (obs, reward, done)
                print("[SUCCESS] execute_pull")
            except Exception as e:
                skill_results['execute_pull'] = str(e)
                print(f"[FAIL] execute_pull: {e}")

        # Try execute_sweep as a generic fallback
        if pick_obj and pick_loc:
            try:
                print(f"Trying execute_sweep on {pick_obj} at {pick_loc}")
                obs, reward, done = execute_sweep(env, task, pick_obj, pick_loc)
                skill_results['execute_sweep'] = (obs, reward, done)
                print("[SUCCESS] execute_sweep")
            except Exception as e:
                skill_results['execute_sweep'] = str(e)
                print(f"[FAIL] execute_sweep: {e}")

        # Try execute_go (move) between two locations
        if len(location_names) >= 2:
            loc_from = location_names[0]
            loc_to = location_names[1]
        elif len(all_names) >= 2:
            loc_from, loc_to = all_names[0], all_names[1]
        else:
            loc_from = loc_to = None

        if loc_from and loc_to:
            try:
                print(f"Trying execute_go from {loc_from} to {loc_to}")
                obs, reward, done = execute_go(env, task, loc_from, loc_to)
                skill_results['execute_go'] = (obs, reward, done)
                print("[SUCCESS] execute_go")
            except Exception as e:
                skill_results['execute_go'] = str(e)
                print(f"[FAIL] execute_go: {e}")

        # Try execute_gripper even though it may be a no-op
        try:
            print("Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            skill_results['execute_gripper'] = (obs, reward, done)
            print("[SUCCESS] execute_gripper")
        except Exception as e:
            skill_results['execute_gripper'] = str(e)
            print(f"[FAIL] execute_gripper: {e}")

        print("===== [Exploration Phase Completed] =====")
        print("Skill Results Summary:")
        for k, v in skill_results.items():
            print(f"  {k}: {v}")

        print("===== Exploration complete: Analyze output above for missing predicates or failed preconditions. =====")

        # END of Exploration Phase
        # The feedback of which operation fails and under what condition should help you diagnose missing predicates.

        # If the plan/oracle is available, you can execute that sequence here. Since you are to focus on exploration due to feedback, above suffices.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
