# run_skeleton_task.py (Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ------------------------------------------------------------------------
        # Exploration Phase: Find Missing Predicate ('drawer-open')
        # ------------------------------------------------------------------------
        #
        # According to feedback, the task involves an issue relating to the 'drawer-open' predicate.
        # Here, we design an exploration sequence using available skills to check/open the drawer.
        #
        # High-level approach:
        #   1. Move to the drawer's location if not already there.
        #   2. Identify/locate drawer and handle objects from 'positions'.
        #   3. Pick the handle (if needed).
        #   4. Attempt to 'pull' (open) the drawer using 'execute_pull'.
        #   5. If already open or after pull, trigger 'drawer-open' predicate in state.

        # --- Identify drawer and handle objects from positions dict ---
        # The keys may look like: 'drawer_1', 'handle_1', etc.
        # We'll try to match objects by checking for substrings.
        drawer_obj = None
        handle_obj = None
        drawer_loc = None
        robot_loc = None
        for k in positions:
            # Try to infer handle/drawer based on keywords
            if "drawer" in k and drawer_obj is None:
                drawer_obj = k
                drawer_loc = positions[k]
            if "handle" in k and handle_obj is None:
                handle_obj = k
        # You may want to adapt these heuristics depending on the positions dictionary

        # For location, you might have a key like "robot", "robot_base", or must infer initial location
        for k in positions:
            if "robot" in k:
                robot_loc = positions[k]
                break

        if drawer_obj is None or handle_obj is None or drawer_loc is None:
            print("[Exploration] ERROR: Could not identify drawer or handle object in environment positions!")
            print("Available objects:", positions)
            return

        print(f"[Exploration] Identified drawer: {drawer_obj} at {drawer_loc}")
        print(f"[Exploration] Identified handle: {handle_obj}")

        # Step 1: Move robot to drawer location if not already there
        try:
            if robot_loc is not None and not np.allclose(robot_loc, drawer_loc, atol=0.2):
                print("[Exploration] Moving robot to drawer location...")
                obs, reward, done = execute_go(env, task, from_location=robot_loc, to_location=drawer_loc)
                if done:
                    print("[Exploration] Stopped after move (robot at drawer).")
                    return
        except Exception as e:
            print("[Exploration] Exception during movement:", e)

        # Step 2: Pick the handle if not already holding
        try:
            print("[Exploration] Attempting to pick handle...")
            obs, reward, done = execute_pick(env, task, handle_obj, drawer_loc)
            if done:
                print("[Exploration] Stopped after pick (handle).")
                return
        except Exception as e:
            print("[Exploration] Exception during pick:", e)

        # Step 3: Try to pull: open the drawer and thus test/add 'drawer-open' predicate
        try:
            print("[Exploration] Attempting to pull (open) drawer via handle...")
            obs, reward, done = execute_pull(env, task, drawer_obj, handle_obj, drawer_loc)
            if done:
                print("[Exploration] Stopped after pull (possibly drawer opened).")
                return
            print("[Exploration] Drawer pull action performed (drawer-open predicate should be in state if successful).")
        except Exception as e:
            print("[Exploration] Exception during pull action:", e)

        # (Optional) Check state/observation for the presence of the 'drawer-open' predicate
        # Normally, the env would have an API or we could inspect obs/state for verification.
        # For now, just log that we've attempted the required exploration to trigger/update the missing predicate.

        # ------------------------------------------------------------------------
        # Main Oracle Plan Execution (If Any)
        # ------------------------------------------------------------------------

        # After exploration, you can continue with the rest of your oracle plan, for example:
        # (The following lines are examples only; fill in the actual task logic if you have a plan.)

        # print("[Task] Continue with oracle plan...")
        # obs, reward, done = execute_place(env, task, obj_to_place, drawer_obj, drawer_loc)
        # if done:
        #     print("[Task] Task ended after place.")
        #     return

        # Further task steps can be appended here as per the goal and plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
