# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Only use the provided, externally-defined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Complete skeleton for running a simulated task with predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # (Position keys depend on RLBench task definition and environment.)

        # === Exploration Phase: Find Missing Predicate ("robot-at drawer-area") ===
        print("[Exploration] Searching for missing predicate or location related to 'drawer-area'...")

        # Assume available location names in the environment
        possible_location_names = list(positions.keys())
        drawer_area_found = None
        for loc_name in possible_location_names:
            if 'drawer' in loc_name or 'drawer-area' in loc_name:
                drawer_area_found = loc_name
                break
        if drawer_area_found is None:
            # Fallback: use the first location if 'drawer-area' cannot be found
            drawer_area_found = possible_location_names[0]
            print(f"[Exploration][Warning] 'drawer-area' not found in object positions. Using default: {drawer_area_found}")
        else:
            print(f"[Exploration] Found likely drawer location: {drawer_area_found}")

        # === Execute Plan to Ensure Predicate "robot-at drawer-area" Holds ===

        # 1. Find where the robot currently is
        robot_area_name = None
        for loc_name in possible_location_names:
            if 'robot' in loc_name or 'start' in loc_name or 'init' in loc_name:
                robot_area_name = loc_name
                break
        if robot_area_name is None:
            # Fallback: use a different area as "from"
            robot_area_name = possible_location_names[0]
        if robot_area_name == drawer_area_found:
            print("[Plan] Robot is already at the drawer area. No movement needed.")
        else:
            try:
                print("[Plan] Moving robot from", robot_area_name, "to", drawer_area_found)
                # Use the corresponding movement skill
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_area_name,
                    to_location=drawer_area_found,
                    timeout=10.0
                )
                print("[Plan] Robot moved to", drawer_area_found)
            except Exception as e:
                print(f"[ERROR] Failed to move robot to drawer area: {str(e)}")

        # 2. (Optional) Additional exploration to ensure the predicate is set in state
        # At this point, `robot-at drawer-area` should hold in the simulated environment.

        # 3. (If desired: You may verify the state via observation or environment; here, we print)
        print(f"[Info] robot-at {drawer_area_found} should now be satisfied in world state.")

        # === Task Complete or Additional Plan Steps Here ===
        # Add further skill executions as necessary to achieve your overall task or goal.
        # For this feedback task, we focus on predicate exploration.

        print("===== Predicate Exploration and Plan Execution Complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()