# run_skeleton_task.py (Completed with Exploration Phase for Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use externally provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # For demonstration, we print available objects and positions
        print("[Info] Object positions:", positions)

        # === EXPLORATION PHASE FOR MISSING PREDICATE ===
        # Feedback indicates the missing predicate involves 'robot-at'.
        # We need to determine whether the environment represents 'robot-at' as:
        #   - robot-at <location>      (per the main domain)
        #   - robot-at <robot> <location> (per exploration domain)
        #
        # We'll explore and check which version works by attempting to perform
        # navigation skills and observing task/environment feedback.

        # List all locations from positions (assuming object positions have associated locations)
        all_locations = set()
        for obj, pos in positions.items():
            if isinstance(pos, dict) and 'location' in pos:
                all_locations.add(pos['location'])
            elif isinstance(pos, str):
                all_locations.add(pos)
            # else: ignore

        if not all_locations:
            # Fallback: try some dummy locations - (update as per your environment knowledge)
            all_locations = {'location_a', 'location_b', 'location_c'}

        location_list = list(all_locations)
        if len(location_list) < 2:
            print("[Warn] Not enough locations to explore predicate!")
            # Can't perform move; skip exploration
            exploration_done = True
        else:
            exploration_done = False

        print("[Exploration] Testing 'robot-at' predicate representations...")
        missing_predicate_found = False
        exploration_result = None

        for idx in range(len(location_list)-1):
            from_loc = location_list[idx]
            to_loc = location_list[idx+1]
            try:
                # Test 1: Call 'execute_go' with (from, to)
                print(f"[Exploration] Trying 'execute_go' between {from_loc} -> {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc,
                    timeout=10.0
                )
                print("[Exploration] Success: 'robot-at' as (location)")
                missing_predicate_form = "robot-at <location>"
                missing_predicate_found = True
                exploration_result = (from_loc, to_loc)
                break
            except TypeError as te:
                # Maybe signature requires (robot, from, to)
                print("[Exploration] TypeError on 'execute_go' with 2 args:", te)
                # Try next style (if skill supports it)
                try:
                    print(f"[Exploration] Trying 'execute_go' with robot argument between {from_loc} -> {to_loc}")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        robot='robot',   # Default robot id/name if required
                        from_location=from_loc,
                        to_location=to_loc,
                        timeout=10.0
                    )
                    print("[Exploration] Success: 'robot-at' as (robot, location)")
                    missing_predicate_form = "robot-at <robot> <location>"
                    missing_predicate_found = True
                    exploration_result = (from_loc, to_loc)
                    break
                except Exception as te2:
                    print("[Exploration] Also failed with robot argument:", te2)
                    continue
            except Exception as e:
                print("[Exploration] Error occurred while testing move:", e)
                continue

        if not missing_predicate_found:
            print("[Exploration] Unable to determine the correct form of 'robot-at' predicate.")
            print("[Exploration] Please verify your domain and skill function signatures.")
        else:
            print(f"[Result] The missing predicate is: {missing_predicate_form}")

        # === MAIN TASK EXECUTION PHASE ===
        # At this point, we've discovered how to use the move skill and thus determined the form of 'robot-at'.
        # You would implement the real oracle plan using the correct skill signatures below.
        #
        # (Pseudo-plan below; replace with actual oracle plan for your task)
        #
        # Example:
        # 1. Move robot to a target location (already done in exploration above)
        # 2. Pick up an object at that location
        # 3. Place object into a drawer, etc.

        if missing_predicate_found:
            from_loc, to_loc = exploration_result
            target_obj = None
            for obj, pos in positions.items():
                # (Assume objects at to_loc are relevant)
                if (isinstance(pos, dict) and pos.get('location', None) == to_loc) or (pos == to_loc):
                    target_obj = obj
                    break
            if target_obj is None:
                target_obj = list(positions.keys())[0]  # fallback: just pick one

            print(f"[Task] Picking up object: {target_obj} at location: {to_loc}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=target_obj,
                    location=to_loc,
                    timeout=10.0
                )
                print(f"[Task] Successfully picked {target_obj}")
            except Exception as e:
                print(f"[Task] Error during pick: {e}")

            # Now try placing object (pseudo drawer and location for example)
            possible_drawers = [obj for obj in positions.keys() if 'drawer' in obj]
            if possible_drawers:
                drawer = possible_drawers[0]
                print(f"[Task] Trying to place {target_obj} into {drawer} at {to_loc}")
                try:
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj=target_obj,
                        drawer=drawer,
                        location=to_loc,
                        timeout=10.0
                    )
                    print(f"[Task] Placed {target_obj} in {drawer}")
                except Exception as e:
                    print(f"[Task] Error during place: {e}")
            else:
                print("[Task] No drawers found in the environment; skipping place.")

            # Add further plan steps here using available skills (execute_pick, execute_place, execute_push, etc.)

        else:
            print("[Task] Cannot execute main plan as exploration failed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
