# run_skeleton_task.py (Complete with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined primitive skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        positions = get_object_positions()
        
        # === EXPLORATION PHASE: Find missing predicate (robot-free) ===
        print("[Exploration] Begin checking for missing predicates or preconditions...")
        
        # The feedback suggests the predicate 'robot-free' might be missing or crucial.
        # Let's use the predefined skills to diagnose it.
        # We'll attempt actions and handle failures to infer what predicate state is missing.
        
        # For demonstration, we'll cycle through all available skills, and try basic actions.
        skill_sequence = [
            ('execute_pick', dict()),    # Parameters will be filled contextually below
            ('execute_place', dict()),
            ('execute_push', dict()),
            ('execute_pull', dict()),
            ('execute_sweep', dict()),
            ('execute_rotate', dict()),
            ('execute_go', dict()),
            ('execute_gripper', dict()),
        ]
        
        # We will try to execute available skills one by one, using environment states
        # This will reveal missing or critical predicates precondition via exception/feedback
        
        # The object and location names expected in the env; adapt as per your scene
        # Try to get one object and location of each type
        sample_object = None
        sample_drawer = None
        sample_location_from = None
        sample_location_to = None
        sample_handle = None

        for key in positions:
            if 'drawer' in key and sample_drawer is None:
                sample_drawer = key
            elif 'handle' in key and sample_handle is None:
                sample_handle = key
            elif 'object' in key or 'ball' in key or 'cube' in key:
                if sample_object is None:
                    sample_object = key
            elif 'loc' in key or 'room' in key or 'table' in key:
                if sample_location_from is None:
                    sample_location_from = key
                elif sample_location_to is None and key != sample_location_from:
                    sample_location_to = key

        # If not enough positions, we simply skip those skills
        # Assemble reasonable arguments for skills
        exploration_attempts = [
            # Try to pick the sample_object if available
            ('execute_pick', {'env': env, 'task': task, 'obj': sample_object, 'location': sample_location_from}),
            # Try placing sample_object into sample_drawer if possible
            ('execute_place', {'env': env, 'task': task, 'obj': sample_object, 'drawer': sample_drawer, 'location': sample_location_from}),
            # Try to open the drawer using the handle if both are available
            ('execute_pull', {'env': env, 'task': task, 'drawer': sample_drawer, 'handle': sample_handle, 'location': sample_location_from}),
            # Try to push-close the drawer
            ('execute_push', {'env': env, 'task': task, 'drawer': sample_drawer, 'location': sample_location_from}),
            # Try basic move action between two locations
            ('execute_go', {'env': env, 'task': task, 'from_location': sample_location_from, 'to_location': sample_location_to}),
            # Try sweep action
            ('execute_sweep', {'env': env, 'task': task, 'obj': sample_object, 'location': sample_location_from}),
            # Try rotate and gripper as well (may do nothing, but try them)
            ('execute_rotate', {'env': env, 'task': task}),
            ('execute_gripper', {'env': env, 'task': task}),
        ]

        for skill, params in exploration_attempts:
            func = globals().get(skill, None)
            if func is None:
                print(f"[Exploration] Skill {skill} not found; skipping...")
                continue
            try:
                print(f"[Exploration] Attempting skill: {skill} with params: {params}")
                func(**params)
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed. Exception: {e}")
                # If failure is related to 'robot-free', output informative message
                if 'robot-free' in str(e):
                    print("[Exploration] Detected missing predicate: (robot-free) is required for this skill.")
                    # Optionally, handle accordingly: e.g., try to reset robot to free state
                # Other exploration feedbacks can be handled similarly
        
        print("[Exploration] Exploration phase complete. If (robot-free) error was detected, ensure future plan provides that state before required skills.")

        # === TASK EXECUTION PHASE ===
        # This is where, in a real system, you'd run a generated oracle plan that
        # now ensures the (robot-free) predicate holds before skills requiring it.

        # For demonstration, print instructions (since no concrete oracle plan is input)
        print("Now ready to execute Oracle Plan. When executing each skill, ensure (robot-free) is true before skills that require it, per feedback and domain.")

        # Example placeholder for future plan (a real plan would execute skills as needed)
        # E.g.,
        #   if is_robot_free():      # ensure robot is free
        #      execute_pick(...)
        #   else:
        #      release_object_or_reset_robot() # some action to make robot free
        
        # (Logic for actual oracle plan execution goes here.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
