# run_skeleton_task.py (Completed With Exploration and Skill Use)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skill functions only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing the simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # The positions dict may contain: {'robot': ..., 'drawer1': ..., 'handle1': ..., ...}
        # This will be used to direct actions

        # ==========================
        # === Exploration Phase ====
        # ==========================
        # Based on the feedback, we know we need to explore the environmental predicate (drawer-closed drawer1)
        # We'll attempt to check (by interaction) whether drawer1 is closed.

        # For this, let's find the robot's starting position and location of drawer1 and handle (if available).
        try:
            robot_pos = positions['robot']
            drawer1_pos = positions['drawer1']
            handle1_pos = positions.get('handle1', None)
        except KeyError as e:
            print(f"[Exploration] Required position not found in 'get_object_positions': {e}")
            return

        # Suppose locations are named and available as IDs, not just xyz; we'll pass them as such. If not, adapt as appropriate for the env.

        # Step 1: Move robot to the drawer (or its handle)
        try:
            print("[Exploration] Moving robot to drawer1 location...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_pos,
                to_location=drawer1_pos
            )
            if done:
                print("[Exploration] Task ended unexpectedly after go to drawer!")
                return
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return
        
        # Step 2: Attempt to identify the state of drawer1 by pulling/opening
        # First, robot may need to pick or hold the handle object before "pull"
        # (Assuming the handle is an object on the floor at the drawer location)
        if handle1_pos is not None:
            try:
                print("[Exploration] Picking up handle1...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name='handle1',
                    location=drawer1_pos
                )
                if done:
                    print("[Exploration] Task ended unexpectedly after execute_pick (handle1)!")
                    return
            except Exception as e:
                print(f"[Exploration] Error during execute_pick (handle1): {e}")
                return

            # Try to pull the drawer open using the handle, checking for lock, closed/open, etc.
            try:
                print("[Exploration] Attempting to pull open drawer1 using handle1...")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name='drawer1',
                    handle_name='handle1',
                    location=drawer1_pos
                )
                if done:
                    print("[Exploration] Task ended after attempt to pull open drawer1!")
                    return
            except Exception as e:
                print(f"[Exploration] Error during execute_pull (drawer1, handle1): {e}")
                # This error may indicate drawer was already open, locked, or closed, or missing predicate
                print("[Exploration] Detected missing or unsatisfied predicate; possible issue: (drawer-closed drawer1) missing predicate; recording this observation.")
                # Exploration phase identifies the missing predicate is likely (drawer-closed drawer1)

        else:
            print("[Exploration] handle1 position not found, cannot attempt execute_pick or execute_pull.")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete: missing predicate is (drawer-closed drawer1)")

        # ===============================
        # === Main Plan Execution Phase ==
        # ===============================

        # Here, you would insert task logic for the main plan using available skills.
        # This part would depend on your oracle plan and environment; here's a mockup:

        # Example:
        # 1) Pick up an object from the floor
        if 'object_1' in positions:
            obj1_pos = positions['object_1']
            try:
                print("[Task] Picking up object_1...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name='object_1',
                    location=obj1_pos
                )
                if done:
                    print("[Task] Task ended after picking up object_1!")
                    return
            except Exception as e:
                print(f"[Task] Error during execute_pick (object_1): {e}")

        # 2) Place the object in the (now open) drawer if possible
        #   (Requires that the drawer is open; may need to pull/open if not)
        try:
            print("[Task] Placing object_1 in drawer1...")
            obs, reward, done = execute_place(
                env,
                task,
                object_name='object_1',
                drawer_name='drawer1',
                location=drawer1_pos
            )
            if done:
                print("[Task] Task ended after placing object_1 in drawer1!")
                return
        except Exception as e:
            print(f"[Task] Error during execute_place (object_1, drawer1): {e}")

        # 3) Close the drawer if required
        try:
            print("[Task] Closing drawer1...")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name='drawer1',
                location=drawer1_pos
            )
            if done:
                print("[Task] Task ended after closing drawer1!")
                return
        except Exception as e:
            print(f"[Task] Error during execute_push (drawer1): {e}")

        # ===========================
        # === End of Plan ===========
        # ===========================

        print("[Task] Plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()