# run_skeleton_task.py (Filled for Exploration/Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # only use these functions directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions['drawer1'], positions['handle1'], ...
        positions = get_object_positions()
        # For robustness, make a list of possibly relevant names based on available info
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_start_loc = 'loc_robot'
        floor_obj_name = 'object1'
        # Attempt retrieving locations; fall back gracefully
        drawer_pos = positions.get(drawer_name, None)
        handle_pos = positions.get(handle_name, None)
        floor_obj_pos = positions.get(floor_obj_name, None)
        robot_pos = positions.get(robot_start_loc, None)

        # The task: Use predefined skills to perform an exploration phase focused on discovering the (drawer-closed drawer1) predicate.

        print("[Exploration] Starting drawer state exploration.")

        # --------------------------------------------------------
        # First: Move to the drawer/handle location (if not already there)

        # NOTE: Without full initial state and object labeling, assume robot should go to drawer1's location.
        # We'll use execute_go primitive for movement.

        try:
            # Find robot's assumed start and drawer's location
            if robot_pos is not None and drawer_pos is not None:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_start_loc,
                    to_location=drawer_name
                )
                print(f"[Action] execute_go from {robot_start_loc} to {drawer_name}")
            else:
                print("[Warning] Could not resolve robot or drawer position. Skipping explicit move.")
        except Exception as e:
            print(f"[Error] execute_go failed: {e}")

        # --------------------------------------------------------
        # Try to pull the drawer's handle to see if it's locked or closed.
        # According to exploration domain, execute_pull may help identify the lock state;
        # and per feedback, the "drawer-closed" predicate may be missing and needs to be checked.

        try:
            # Must pick handle first before pulling (from PDDL, must be holding handle)
            if handle_name in positions:
                handle_loc = drawer_name  # assume handle is co-located with drawer
                # 1) Pick handle
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=handle_name,
                    loc=drawer_name
                )
                print(f"[Action] execute_pick {handle_name} at {drawer_name}")

                # 2) Attempt execute_pull with handle
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer_name,
                    obj=handle_name,
                    loc=drawer_name
                )
                print(f"[Action] execute_pull {drawer_name} with {handle_name}")

            else:
                print("[Warning] Could not find handle position for execute_pull, skipping.")

        except Exception as e:
            print(f"[Error] Exploration - execute_pick/execute_pull failed: {e}")

        # --------------------------------------------------------
        # Based on feedback, if the pull fails, we infer the drawer was not openable - likely due to a missing predicate regarding the drawer being closed.
        # The missing predicate (drawer-closed drawer1) informs the plan's refinement.

        print("[Exploration] Checked for missing predicate: likely requires (drawer-closed drawer1) in initial state.")

        # Additional exploration (optional): check if robot can place or open drawer, or try further action with drawer, as needed in a complete pipeline.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
