# run_skeleton_task.py (Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with missing predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing the simulation (optional)
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # You may want to print positions or investigate their content:
        # print("[Debug] Object Positions:", positions)

        # === Exploration Phase: Identify Missing Predicates ===
        #
        # Based on feedback that the planner times out,
        # and based on exploration domain knowledge,
        # we perform exploration using available primitives.
        #
        # Our method is to use as many primitives ("skills") as possible
        # to try to cause an effect that would indicate a missing predicate.
        #
        # We will attempt basic actions and check for 
        # error signals or changes in the environment.
        #
        # Since we are NOT allowed to define new skills,
        # and since the missing predicate may be required for plan progression,
        # we will use the available skills to test for what's possible.

        # List of available skills:
        skill_names = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]
        
        # Print exploration info
        print("[Exploration] Starting exploration of missing predicates using available skills.")

        # Attempt to use each skill in a generic fashion to see if any preconditions fail
        # For demonstration, let's try to use each skill one by one with generic arguments appropriately fetched from environment info
        any_action_succeeded = False

        # For exploration, we try to use the skill functions in a safe way
        # We'll retrieve at least one "object", one "drawer", and one "location" from positions, if possible
        objects = [obj for obj in positions if positions[obj].get('type') == 'object']
        drawers = [obj for obj in positions if positions[obj].get('type') == 'drawer']
        locations = [obj for obj in positions if positions[obj].get('type') == 'location']
        handles = [obj for obj in positions if positions[obj].get('type') == 'handle']

        # Fallback: if no type info, try to categorize based on name (user should improve get_object_positions for exact types)
        if not objects:
            objects = [name for name in positions if 'obj' in name or 'ball' in name or 'cube' in name]
        if not drawers:
            drawers = [name for name in positions if 'drawer' in name]
        if not locations:
            locations = [name for name in positions if 'loc' in name or 'room' in name or 'table' in name or 'floor' in name]
        if not handles:
            handles = [name for name in positions if 'handle' in name]

        # Select representative items (if available)
        obj = objects[0] if objects else None
        drawer = drawers[0] if drawers else None
        location = locations[0] if locations else None
        handle = handles[0] if handles else None

        # Try skill usage according to expected parameters from domain
        # 1. Try execute_pick (object, location)
        try:
            if obj and location:
                print(f"[Exploration] Trying execute_pick(obj={obj}, location={location})")
                obs, reward, done = execute_pick(env, task, obj, location)
                any_action_succeeded = True
            else:
                print("[Exploration] Insufficient objects/locations for execute_pick")
        except Exception as e:
            print("[Exploration] execute_pick failed:", e)

        # 2. Try execute_place (object, drawer, location)
        try:
            if obj and drawer and location:
                print(f"[Exploration] Trying execute_place(obj={obj}, drawer={drawer}, location={location})")
                obs, reward, done = execute_place(env, task, obj, drawer, location)
                any_action_succeeded = True
            else:
                print("[Exploration] Insufficient objects/drawers/locations for execute_place")
        except Exception as e:
            print("[Exploration] execute_place failed:", e)

        # 3. Try execute_push (drawer, location)
        try:
            if drawer and location:
                print(f"[Exploration] Trying execute_push(drawer={drawer}, location={location})")
                obs, reward, done = execute_push(env, task, drawer, location)
                any_action_succeeded = True
            else:
                print("[Exploration] Insufficient drawers/locations for execute_push")
        except Exception as e:
            print("[Exploration] execute_push failed:", e)

        # 4. Try execute_pull (drawer, handle, location)
        try:
            if drawer and handle and location:
                print(f"[Exploration] Trying execute_pull(drawer={drawer}, handle={handle}, location={location})")
                obs, reward, done = execute_pull(env, task, drawer, handle, location)
                any_action_succeeded = True
            else:
                print("[Exploration] Insufficient drawers/handles/locations for execute_pull")
        except Exception as e:
            print("[Exploration] execute_pull failed:", e)

        # 5. Try execute_sweep (object, location)
        try:
            if obj and location:
                print(f"[Exploration] Trying execute_sweep(obj={obj}, location={location})")
                obs, reward, done = execute_sweep(env, task, obj, location)
                any_action_succeeded = True
            else:
                print("[Exploration] Insufficient objects/locations for execute_sweep")
        except Exception as e:
            print("[Exploration] execute_sweep failed:", e)

        # 6. Try execute_go (from, to)
        try:
            if len(locations) >= 2:
                from_loc, to_loc = locations[0], locations[1]
                print(f"[Exploration] Trying execute_go(from={from_loc}, to={to_loc})")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                any_action_succeeded = True
            else:
                print("[Exploration] Not enough locations for execute_go")
        except Exception as e:
            print("[Exploration] execute_go failed:", e)

        # 7. Try execute_rotate (if available)
        try:
            # Not enough domain info, so we just show attempt
            print(f"[Exploration] Trying execute_rotate()")
            obs, reward, done = execute_rotate(env, task)
            any_action_succeeded = True
        except Exception as e:
            print("[Exploration] execute_rotate failed:", e)

        # 8. Try execute_gripper (if available)
        try:
            print(f"[Exploration] Trying execute_gripper()")
            obs, reward, done = execute_gripper(env, task)
            any_action_succeeded = True
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        # Report if anything succeeded (dummy exploration)
        if any_action_succeeded:
            print("[Exploration] At least one skill executed without crashing; check logs/environment for predicate condition changes.")
        else:
            print("[Exploration] No available skill succeeded; determine missing predicates based on error feedback above.")

        # === Oracle Plan Execution Phase ===
        #
        # Here, if an oracle plan is known, step through it and use the relevant skill functions.
        # This section is left generic -- in practice, you would parse a plan and
        # sequentially map PDDL actions to their Python skill functions.
        #
        # For example:
        # plan = [('execute_go', from, to), ('execute_pick', obj, loc), ...]
        # for step in plan:
        #     action_name = step[0]
        #     params = step[1:]
        #     try:
        #         skill_fn = globals()[action_name]
        #         obs, reward, done = skill_fn(env, task, *params)
        #         if done:
        #             print("[TASK] Plan execution completed.")
        #             break
        #     except Exception as e:
        #         print(f"[TASK][ERROR] Failed at step {step}: {e}")
        #
        # In reality, you must update the above with the real plan and parameters.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()