# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails due to a missing predicate (e.g., lock-known).

        # Get all objects, drawers, and locations from positions
        objects = []
        drawers = []
        locations = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'loc' in name or 'location' in name:
                locations.append(name)
            else:
                objects.append(name)

        # For this exploration, we will:
        # 1. Try to pick up each object on the floor at its location.
        # 2. Try to open each drawer by pulling its handle.
        # 3. Try to place an object in a drawer.
        # 4. Try to move between locations.

        # We will catch exceptions and print them to infer missing predicates.

        # 1. Try to pick up each object
        for obj in objects:
            try:
                # Find the location of the object
                obj_pos = positions[obj]
                # Find the closest location name (if available)
                obj_loc = None
                for loc in locations:
                    if np.linalg.norm(np.array(positions[loc]) - np.array(obj_pos)) < 0.5:
                        obj_loc = loc
                        break
                if obj_loc is None and locations:
                    obj_loc = locations[0]
                print(f"[Exploration] Attempting to pick up {obj} at {obj_loc}")
                obs, reward, done = execute_pick(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully picked up {obj}")
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}: {e}")

        # 2. Try to open each drawer by pulling its handle
        for drawer in drawers:
            try:
                # Find the handle for the drawer (assume handle is named 'handle_of_' + drawer)
                handle_name = f"handle_of_{drawer}"
                if handle_name not in positions:
                    print(f"[Exploration] No handle found for {drawer}")
                    continue
                handle_pos = positions[handle_name]
                # Find the closest location to the handle
                handle_loc = None
                for loc in locations:
                    if np.linalg.norm(np.array(positions[loc]) - np.array(handle_pos)) < 0.5:
                        handle_loc = loc
                        break
                if handle_loc is None and locations:
                    handle_loc = locations[0]
                print(f"[Exploration] Attempting to pick up handle {handle_name} at {handle_loc}")
                try:
                    obs, reward, done = execute_pick(env, task, handle_name, handle_loc)
                except Exception as e:
                    print(f"[Exploration] Failed to pick up handle {handle_name}: {e}")
                    continue
                print(f"[Exploration] Attempting to pull {drawer} using {handle_name} at {handle_loc}")
                try:
                    obs, reward, done = execute_pull(env, task, drawer, handle_name, handle_loc)
                    if done:
                        print(f"[Exploration] Successfully pulled {drawer}")
                except Exception as e:
                    print(f"[Exploration] Failed to pull {drawer}: {e}")
            except Exception as e:
                print(f"[Exploration] Error with drawer {drawer}: {e}")

        # 3. Try to place each object in each drawer (if possible)
        for obj in objects:
            for drawer in drawers:
                try:
                    # Find a location for the drawer
                    drawer_pos = positions[drawer]
                    drawer_loc = None
                    for loc in locations:
                        if np.linalg.norm(np.array(positions[loc]) - np.array(drawer_pos)) < 0.5:
                            drawer_loc = loc
                            break
                    if drawer_loc is None and locations:
                        drawer_loc = locations[0]
                    print(f"[Exploration] Attempting to place {obj} in {drawer} at {drawer_loc}")
                    obs, reward, done = execute_place(env, task, obj, drawer, drawer_loc)
                    if done:
                        print(f"[Exploration] Successfully placed {obj} in {drawer}")
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} in {drawer}: {e}")

        # 4. Try to move between all pairs of locations
        for from_loc in locations:
            for to_loc in locations:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    if done:
                        print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                except Exception as e:
                    print(f"[Exploration] Failed to move from {from_loc} to {to_loc}: {e}")

        # 5. Try to push (close) each drawer
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to push (close) {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] Successfully pushed (closed) {drawer}")
                except Exception as e:
                    print(f"[Exploration] Failed to push (close) {drawer} at {loc}: {e}")

        # 6. Try to sweep each object
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}")
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")

        # 7. Try to use gripper
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Successfully used gripper")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Review above output for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()