# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (on-floor dice2)
        # The goal is to explore the environment to determine which predicate is missing.
        # We will use available skills to interact with objects and observe state changes.

        # Step 1: List all objects and their positions
        print("[Exploration] Object positions:", positions)
        # For this example, let's assume 'dice2' is one of the objects
        dice2_name = None
        for obj_name in positions:
            if 'dice2' in obj_name:
                dice2_name = obj_name
                break
        if dice2_name is None:
            print("[Exploration] dice2 not found in object positions. Exploration aborted.")
            return

        dice2_pos = positions[dice2_name]
        print(f"[Exploration] Found dice2 at position: {dice2_pos}")

        # Step 2: Move robot to the location of dice2
        # Assume robot's current location is available in obs or positions
        # For this example, we try to find the robot's current location
        robot_location = None
        for key in positions:
            if 'robot' in key:
                robot_location = positions[key]
                break
        if robot_location is None:
            # Fallback: try to get from obs
            if hasattr(obs, 'robot_position'):
                robot_location = obs.robot_position
            else:
                print("[Exploration] Robot location not found. Exploration aborted.")
                return

        # Step 3: Use execute_go to move to dice2's location
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {dice2_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=dice2_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Robot moved to dice2 location.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # Step 4: Try to pick up dice2 using execute_pick
        try:
            print(f"[Exploration] Attempting to pick up {dice2_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=dice2_name,
                location=dice2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Exploration] execute_pick completed.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")
            return

        # Step 5: Observe the state after picking up dice2
        # The missing predicate is likely related to the state of dice2 (e.g., on-floor, holding, etc.)
        # Print out the relevant state information
        try:
            obs = task.get_observation()
            if hasattr(obs, 'holding_objects'):
                print("[Exploration] Currently holding objects:", obs.holding_objects)
            if hasattr(obs, 'object_states'):
                print("[Exploration] Object states:", obs.object_states)
        except Exception as e:
            print(f"[Exploration] Error retrieving observation: {e}")

        # Step 6: Conclude exploration
        print("[Exploration] Exploration phase complete. Check logs for missing predicate information.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
