# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate or Object ===
        # Feedback: "dice1" is referenced but not present in the object list.
        # We need to check for missing objects and predicates before executing the plan.

        # Let's assume the objects and drawers are listed in positions
        # For demonstration, we expect at least two dice and a drawer handle
        required_objects = ['dice1', 'dice2']
        drawer_name = None
        handle_name = None

        # Try to find dice and drawer/handle in the positions dictionary
        available_objects = list(positions.keys())
        missing_objects = [obj for obj in required_objects if obj not in available_objects]

        # Try to find a drawer and its handle
        for obj in available_objects:
            if 'drawer' in obj and 'handle' not in obj:
                drawer_name = obj
            if 'handle' in obj:
                handle_name = obj

        # === Safety and Existence Checks ===
        if missing_objects:
            print(f"[Exploration] Missing objects in environment: {missing_objects}")
            print("[Exploration] Aborting task due to missing required objects.")
            return

        if drawer_name is None or handle_name is None:
            print("[Exploration] Drawer or handle not found in environment objects.")
            print("[Exploration] Aborting task due to missing drawer or handle.")
            return

        # === Exploration: Check Drawer State (open/closed/locked/unlocked) ===
        # We need to ensure the drawer is unlocked and closed before opening
        # Since we don't have direct predicates, we can try to open the drawer and catch errors

        # Move robot to drawer location if not already there
        robot_location = None
        for obj in available_objects:
            if 'robot' in obj:
                robot_location = positions[obj]
                break

        drawer_location = positions[drawer_name]
        handle_location = positions[handle_name]

        # If robot location is not at drawer, move there
        # We'll use execute_go (from, to)
        # For simplicity, assume location names are the keys in positions
        # If robot location is not known, skip move
        if robot_location is not None and robot_location != drawer_location:
            try:
                print(f"[Task] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                if done:
                    print("[Task] Task ended after move!")
                    return
            except Exception as e:
                print(f"[Error] Exception during execute_go: {e}")

        # === Exploration: Try to pick the handle ===
        try:
            print(f"[Task] Attempting to pick handle: {handle_name} at {drawer_location}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
            if done:
                print("[Task] Task ended after picking handle!")
                return
        except Exception as e:
            print(f"[Error] Exception during execute_pick (handle): {e}")
            print("[Exploration] Handle may not be pickable or already held.")

        # === Exploration: Try to pull (open) the drawer ===
        try:
            print(f"[Task] Attempting to pull (open) drawer: {drawer_name} with handle: {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            if done:
                print("[Task] Task ended after pulling drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during execute_pull: {e}")
            print("[Exploration] Drawer may be locked or already open.")

        # === Main Task: Put dice in the drawer ===
        for dice in required_objects:
            # Check if dice is present
            if dice not in positions:
                print(f"[Task] {dice} not found in environment, skipping.")
                continue

            dice_location = positions[dice]

            # Move to dice location if not already there
            if robot_location != dice_location:
                try:
                    print(f"[Task] Moving robot from {drawer_location} to {dice_location} to pick {dice}")
                    obs, reward, done = execute_go(env, task, drawer_location, dice_location)
                    if done:
                        print("[Task] Task ended after move to dice!")
                        return
                    robot_location = dice_location
                except Exception as e:
                    print(f"[Error] Exception during execute_go (to dice): {e}")

            # Pick the dice
            try:
                print(f"[Task] Picking up {dice} at {dice_location}")
                obs, reward, done = execute_pick(env, task, dice, dice_location)
                if done:
                    print("[Task] Task ended after picking dice!")
                    return
            except Exception as e:
                print(f"[Error] Exception during execute_pick (dice): {e}")

            # Move back to drawer
            if robot_location != drawer_location:
                try:
                    print(f"[Task] Moving robot from {dice_location} to {drawer_location} to place {dice}")
                    obs, reward, done = execute_go(env, task, dice_location, drawer_location)
                    if done:
                        print("[Task] Task ended after move to drawer!")
                        return
                    robot_location = drawer_location
                except Exception as e:
                    print(f"[Error] Exception during execute_go (to drawer): {e}")

            # Place the dice in the drawer
            try:
                print(f"[Task] Placing {dice} in {drawer_name} at {drawer_location}")
                obs, reward, done = execute_place(env, task, dice, drawer_name, drawer_location)
                if done:
                    print("[Task] Task ended after placing dice!")
                    return
            except Exception as e:
                print(f"[Error] Exception during execute_place: {e}")

        # === Optionally, close the drawer ===
        try:
            print(f"[Task] Attempting to push (close) drawer: {drawer_name}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            if done:
                print("[Task] Task ended after pushing drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during execute_push: {e}")

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()