# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from observation and feedback) ===
        dice_names = ['dice1', 'dice2']
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_init_pose = 'ready-pose'
        floor_name = 'floor'
        drawer_area = 'drawer-area'

        # === Exploration Phase: Check for Missing Predicate (e.g., Drawer Accessibility) ===
        # The feedback suggests we need to ensure the drawer is accessible and not full before placing dice.
        # We'll use the available skills to check the state and perform safe placement.

        # 1. Move robot to ready-pose if not already there
        try:
            robot_pos = positions.get('robot', None)
            if robot_pos is not None and robot_pos != robot_init_pose:
                obs, reward, done = execute_go(env, task, robot_pos, robot_init_pose)
        except Exception as e:
            print("[Exploration] Error moving robot to ready-pose:", e)

        # 2. Safety/Exploration: Ensure drawer is open and not full
        # (In this domain, drawer1 is open and empty at init, but we check for robustness)
        # If not open, try to open it using handle
        try:
            # Check if drawer is open (simulate by checking positions or state if available)
            drawer_state = positions.get(f'{drawer_name}_state', 'open')  # fallback to 'open'
            if drawer_state != 'open':
                # Pick handle
                obs, reward, done = execute_pick(env, task, handle_name, floor_name)
                # Pull to open drawer
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_init_pose)
        except Exception as e:
            print("[Exploration] Error ensuring drawer is open:", e)

        # 3. Safety: Check if drawer is full (simulate by checking positions or state if available)
        try:
            drawer_full = positions.get(f'{drawer_name}_full', False)
            if drawer_full:
                print("[Exploration] Drawer is full, cannot place dice.")
                return
        except Exception as e:
            print("[Exploration] Error checking drawer fullness:", e)

        # 4. For each dice, pick and place into the open drawer
        for dice in dice_names:
            try:
                # Check if dice is on the floor
                dice_on_floor = positions.get(f'{dice}_on_floor', True)
                if not dice_on_floor:
                    print(f"[Task] {dice} is not on the floor, skipping.")
                    continue

                # Pick dice
                obs, reward, done = execute_pick(env, task, dice, floor_name)
                # Place dice in drawer
                obs, reward, done = execute_place(env, task, dice, drawer_name, robot_init_pose)

                # After placing, check if drawer is now full (simulate)
                drawer_full = positions.get(f'{drawer_name}_full', False)
                if drawer_full:
                    print(f"[Task] Drawer became full after placing {dice}.")
                    break

            except Exception as e:
                print(f"[Task] Error handling {dice}:", e)

        # 5. (Optional) Push drawer closed if required (not required by goal, but safe)
        try:
            # If drawer should be closed at the end, uncomment below:
            # obs, reward, done = execute_push(env, task, drawer_name, robot_init_pose)
            pass
        except Exception as e:
            print("[Task] Error closing drawer:", e)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()