# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails due to a missing predicate (e.g., lock-known).

        # Get all objects, drawers, and locations from positions
        objects = []
        drawers = []
        locations = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'loc' in name or 'location' in name:
                locations.append(name)
            else:
                objects.append(name)

        # For exploration, we will:
        # 1. Try to pick up each object on the floor at its location.
        # 2. Try to open each drawer by pulling its handle.
        # 3. Try to place an object in a drawer.
        # 4. Try to move between locations.

        # We will catch exceptions and print out which action failed, which may indicate a missing predicate.

        # 1. Try to pick up each object
        for obj in objects:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to pick up object: {obj} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}: {e}")

        # 2. Try to open each drawer by pulling its handle
        for drawer in drawers:
            # Find handle for this drawer
            handle = None
            for obj in objects:
                # Assume handle-of relation is encoded in object name or positions
                if f"handle_{drawer}" in obj or f"{drawer}_handle" in obj or "handle" in obj:
                    handle = obj
                    break
            if handle is None:
                continue
            try:
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to pick up handle: {handle} at {positions[handle]}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle,
                    positions[handle]
                )
                if done:
                    print(f"[Exploration] Picked up handle {handle} for drawer {drawer}")
                print(f"[Exploration] Attempting to pull open drawer: {drawer} with handle {handle}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pulled open {drawer}")
            except Exception as e:
                print(f"[Exploration] Failed to pull open {drawer} with handle {handle}: {e}")

        # 3. Try to place an object in a drawer
        for drawer in drawers:
            for obj in objects:
                try:
                    drawer_pos = positions[drawer]
                    print(f"[Exploration] Attempting to place {obj} in {drawer} at {drawer_pos}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        drawer_pos
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} in {drawer}")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} in {drawer}: {e}")

        # 4. Try to move between locations
        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
            except Exception as e:
                print(f"[Exploration] Failed to move from {from_loc} to {to_loc}: {e}")

        # 5. Try to push drawers closed
        for drawer in drawers:
            try:
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to push close drawer: {drawer} at {drawer_pos}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pushed closed {drawer}")
            except Exception as e:
                print(f"[Exploration] Failed to push close {drawer}: {e}")

        # 6. Try to sweep each object
        for obj in objects:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to sweep object: {obj} at {obj_pos}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}")
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 7. Try to use gripper
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check above for any failed actions indicating missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()