# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We'll attempt to use the available skills to trigger any observable state changes.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For exploration, we will try to use each skill on each object/location if possible.
        # This is a generic exploration loop to help discover missing predicates.

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']
        handle_names = [name for name in positions if positions[name].get('type', '') == 'handle']

        # Fallback: If type info is not available, use all keys as objects/locations
        if not object_names:
            object_names = [name for name in positions if 'object' in name]
        if not drawer_names:
            drawer_names = [name for name in positions if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions if 'location' in name or 'room' in name or 'pos' in name]
        if not handle_names:
            handle_names = [name for name in positions if 'handle' in name]

        # If still empty, just use all keys
        if not object_names:
            object_names = list(positions.keys())
        if not location_names:
            location_names = list(positions.keys())

        # Try to move to each location
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] Task ended after execute_go from {from_loc} to {to_loc}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_go failed: {e}")

        # Try to pick each object at each location
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Task ended after execute_pick on {obj} at {loc}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # Try to place each object in each drawer at each location
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_place of {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] Task ended after execute_place of {obj} in {drawer} at {loc}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {e}")

        # Try to pull each drawer handle at each location
        for drawer in drawer_names:
            for handle in handle_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_pull on {drawer} with {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] Task ended after execute_pull on {drawer} with {handle} at {loc}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed: {e}")

        # Try to push each drawer at each location
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] Task ended after execute_push on {drawer} at {loc}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

        # Try to sweep each object at each location
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Task ended after execute_sweep on {obj} at {loc}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {e}")

        # Try to rotate (if available)
        try:
            print(f"[Exploration] Trying execute_rotate")
            obs, reward, done = execute_rotate(env, task)
            if done:
                print(f"[Exploration] Task ended after execute_rotate")
                return
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {e}")

        # Try to use gripper (if available)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Task ended after execute_gripper")
                return
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Completed all available skill explorations. If the missing predicate is discovered, update the domain/problem accordingly.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()