# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will attempt to use available skills to explore the state of drawer1.

        # Assumptions for object names and locations
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = 'loc_robot'
        drawer_location = 'loc_drawer'
        # These names should match your environment's naming conventions.
        # If not, adjust accordingly.

        # Try to get positions for drawer and handle
        try:
            drawer_pos = positions[drawer_name]
        except KeyError:
            print(f"[Exploration] Drawer '{drawer_name}' not found in positions.")
            drawer_pos = None
        try:
            handle_pos = positions[handle_name]
        except KeyError:
            print(f"[Exploration] Handle '{handle_name}' not found in positions.")
            handle_pos = None

        # Step 1: Move robot to drawer location if not already there
        try:
            # If robot location is known, move to drawer location
            if robot_location in positions and drawer_location in positions:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=drawer_location,
                    max_steps=100
                )
                print(f"[Exploration] Robot moved from {robot_location} to {drawer_location}.")
            else:
                print("[Exploration] Robot or drawer location not found in positions.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # Step 2: Try to pick the handle (if handle exists)
        try:
            if handle_name in positions:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=handle_name,
                    location=drawer_location,
                    max_steps=100
                )
                print(f"[Exploration] Picked up handle '{handle_name}'.")
            else:
                print(f"[Exploration] Handle '{handle_name}' not found in positions.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_pick: {e}")

        # Step 3: Try to pull the drawer open (this may reveal if it's closed/locked)
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location,
                max_steps=100
            )
            print(f"[Exploration] Attempted to pull drawer '{drawer_name}' using handle '{handle_name}'.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_pull: {e}")

        # Step 4: If drawer is open, try to push it closed (to ensure (drawer-closed ...))
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_location,
                max_steps=100
            )
            print(f"[Exploration] Attempted to push drawer '{drawer_name}' closed.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_push: {e}")

        # Step 5: Exploration complete, check for missing predicate
        # (In a real system, you would now check the state for (drawer-closed drawer1))
        print("[Exploration] Exploration phase complete. Check logs or state for (drawer-closed drawer1).")

        # === Main Task Plan (if any) ===
        # Insert your oracle plan here, using only the available skills.
        # For example:
        # obs, reward, done = execute_pick(env, task, object_name='obj1', location='loc1', max_steps=100)
        # obs, reward, done = execute_place(env, task, object_name='obj1', drawer_name='drawer1', location='loc1', max_steps=100)
        # etc.

        # TODO: Insert main plan steps here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
