# run_skeleton_task.py (Fully Generic Skeleton)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills, do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ===============================
        # === Exploration Phase Begins ===
        # ===============================
        # The domain or planner failed to generate a plan, likely due to a missing predicate.
        # According to the exploration knowledge, we perform a systematic exploration to determine which object/drawer has an unknown ('missing') property/state.

        # Use the available skills to explore: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper
        # In the absence of an oracle plan, we use exploration skills to check the environment for missing predicates.

        # For demonstration, we will:
        # - Iterate through all positions/objects in the environment
        # - For each, attempt to perform one or several actions that could reveal missing properties.
        # - If an action fails (e.g., cannot open a drawer), we deduce that a predicate such as 'drawer-locked' or 'drawer-unlocked' may be missing, and we log this.

        # NOTE: You MUST NOT implement or call any function that is not in skill_code.
        # Assume object_types help us decide: objects, drawers, handles, locations

        # Extract environment state information, with fallback handling
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        handles = positions.get('handles', [])
        locations = positions.get('locations', [])
        robot_pos = positions.get('robot', None)

        print("[Exploration] Environment objects:", objects)
        print("[Exploration] Drawers:", drawers)
        print("[Exploration] Handles:", handles)
        print("[Exploration] Locations:", locations)
        print("[Exploration] Robot position:", robot_pos)

        # Attempt to identify missing predicates through action trial/exploration
        exploration_success = False
        missing_predicate_found = False

        # Loop over all drawers and try to pull handles to check lock state
        for drawer in drawers:
            # Find associated handle
            handle = None
            for h in handles:
                if h.get('drawer') == drawer['name']:
                    handle = h
                    break
            if not handle:
                continue  # Cannot find handle

            # Move robot to drawer location if not already there
            drawer_location = drawer['location']
            if robot_pos is not None and robot_pos != drawer_location:
                try:
                    print(f"[Exploration] Moving robot from {robot_pos} to {drawer_location}")
                    obs, reward, done = execute_go(env, task, robot_pos, drawer_location)
                    robot_pos = drawer_location
                except Exception as e:
                    print(f"[Exploration] ERROR: execute_go to {drawer_location} failed: {e}")

            # Try to pick up handle
            try:
                print(f"[Exploration] Picking (grasping) handle {handle['name']} at {drawer_location}")
                obs, reward, done = execute_pick(env, task, handle['name'], drawer_location)
            except Exception as e:
                print(f"[Exploration] ERROR: execute_pick handle {handle['name']} failed: {e}")

            # Try to pull the drawer to see if it's locked/unlocked
            try:
                print(f"[Exploration] Pulling drawer {drawer['name']} using handle {handle['name']} at {drawer_location}")
                obs, reward, done = execute_pull(env, task, drawer['name'], handle['name'], drawer_location)
                print(f"[Exploration] Drawer {drawer['name']} pulled successfully: probably unlocked.")
                exploration_success = True
            except Exception as e:
                print(f"[Exploration] ERROR: execute_pull on drawer {drawer['name']} failed (maybe locked or missing predicate): {e}")
                missing_predicate_found = True

        # Sweep through all objects on the floor to check if on-floor property is missing
        for obj in objects:
            obj_loc = obj.get('location')
            try:
                print(f"[Exploration] Sweeping object {obj['name']} at {obj_loc}")
                obs, reward, done = execute_sweep(env, task, obj['name'], obj_loc)
            except Exception as e:
                print(f"[Exploration] ERROR: execute_sweep on object {obj['name']} failed: {e}")

        # Attempt to pick and place an object into a drawer to check fullness predicates
        for obj in objects:
            obj_loc = obj.get('location')
            if len(drawers) == 0:
                break
            target_drawer = drawers[0]
            drawer_loc = target_drawer['location']
            try:
                print(f"[Exploration] Picking object {obj['name']} at {obj_loc}")
                obs, reward, done = execute_pick(env, task, obj['name'], obj_loc)
            except Exception as e:
                print(f"[Exploration] ERROR: execute_pick on object {obj['name']} failed: {e}")
            try:
                print(f"[Exploration] Placing object {obj['name']} into drawer {target_drawer['name']} at {drawer_loc}")
                obs, reward, done = execute_place(env, task, obj['name'], target_drawer['name'], drawer_loc)
            except Exception as e:
                print(f"[Exploration] ERROR: execute_place into drawer {target_drawer['name']} failed (maybe drawer is not open or full/empty predicate missing): {e}")

        if missing_predicate_found:
            print("[Exploration] -> Missing predicate(s) likely detected in domain model. See above errors.")
        elif exploration_success:
            print("[Exploration] -> Core properties interacted as expected. No lock/full/etc. predicate explicitly missing.")
        else:
            print("[Exploration] -> No meaningful exploration was possible. Environment/skill mapping may be incomplete.")

        # ===============================
        # === End of Exploration Phase ===
        # ===============================

        # Insert your actual oracle plan/execution control logic here afterwards, as needed.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
