# run_skeleton_task.py (Fully Generic Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Get current object positions and state ===
        positions = get_object_positions()

        # === Exploration phase: Identify missing predicates ===
        # Feedback says: (drawer-open drawer3) was relevant/missing
        # Exploration: We need to determine the lock state or openness of drawer3
        # Strategy: Use available skills to gather this knowledge.
        # Skill list: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        #              'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # 
        # Since handle and drawer objects may be named, we'll search for appropriate object names

        object_names = list(positions.keys())

        # Identify the drawer of interest (the one in feedback): drawer3
        target_drawer_name = None
        for obj in object_names:
            if "drawer3" in obj.lower():
                target_drawer_name = obj
                break
        # For handle, try to find the corresponding handle for drawer3
        handle_name = None
        for obj in object_names:
            if "handle" in obj.lower() and "drawer3" in obj.lower():
                handle_name = obj
                break
        if not handle_name:
            # If not found, pick any handle (fallback, may require environment knowledge)
            for obj in object_names:
                if "handle" in obj.lower():
                    handle_name = obj
                    break

        # Pick a robot start location and drawer3 location by types
        locations = []
        for obj in object_names:
            if any(s in obj.lower() for s in ["table", "desk", "loc", "stand", "drawer"]):
                locations.append(obj)
        # Try to get the robot's current location
        robot_location = None
        for obj in positions:
            # The robot's current location might be a known key (e.g., 'robot', 'robot_base', etc.)
            if "robot" in obj.lower() and positions[obj] is not None:
                robot_location = obj
                break
        # Fallback: pick any location
        if not robot_location and locations:
            robot_location = locations[0]

        # Try to get drawer3's location (could be its own name, or a nearby place)
        drawer3_location = target_drawer_name if target_drawer_name else (locations[-1] if locations else None)

        # --- Exploration: Try skills that reveal the state of the drawer ---
        # For a closed or locked drawer, only opening or pulling would cause new information
        # Approach:
        # - Move to drawer3's location if needed
        # - Try execute_pull on the handle (if the handle is associated)
        # Since we can't redefine skills, call them as-is
        # We'll try the actions that could reveal missing predicates

        try:
            # If robot not at drawer3, go there
            if robot_location and drawer3_location and robot_location != drawer3_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer3_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer3_location)
                robot_location = drawer3_location
                if done:
                    print("[Exploration] Task ended during movement!")
                    return

            # Pick the handle of drawer3 (try picking if not already holding)
            if handle_name:
                print(f"[Exploration] Picking handle {handle_name} at {drawer3_location}")
                obs, reward, done = execute_pick(env, task, handle_name, drawer3_location)
                if done:
                    print("[Exploration] Task ended after picking handle!")
                    return

                # Now try pulling the handle to attempt to open (and learn about lock!)
                print(f"[Exploration] Pulling {target_drawer_name} using {handle_name} at {drawer3_location}")
                obs, reward, done = execute_pull(env, task, target_drawer_name, handle_name, drawer3_location)
                if done:
                    print("[Exploration] Task ended after pulling drawer!")
                    return

            else:
                # If no handle found, try to directly open (pull) the drawer
                if target_drawer_name:
                    print(f"[Exploration] Pulling {target_drawer_name} directly at {drawer3_location}")
                    obs, reward, done = execute_pull(env, task, target_drawer_name, None, drawer3_location)
                    if done:
                        print("[Exploration] Task ended after pulling drawer!")
                        return
                else:
                    print("[Exploration] No handle or drawer3 identified; skipping pull.")

        except Exception as e:
            print(f"[Exploration] Exception during exploration: {e}")
        
        print("[Exploration] Exploration phase finished.")

        # ================== PLAN EXECUTION PHASE =======================
        # Plan execution would proceed here (omitted, as focus is on missing predicate exploration).
        # Example:
        # print("[Task] Manipulating object_1 at:", object_1_pos)
        # obs, reward, done = execute_pick(env, task, ...)
        # if done: return

        # TODO: Control flow to solve main task plan as needed.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()