# run_skeleton_task.py (Filled with Exploration Phase for Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Skeleton for running a task and exploring for missing predicates/skills.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Optionally initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording, if needed
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Exploration Phase: Diagnose the missing predicate via actions
        print("[Exploration] Beginning exploration to diagnose missing predicates.")

        # Typical exploration steps (hypothetical, to trigger different preconditions and effects):

        # --- Example exploration structure ---
        # 1. Attempt to move the robot between locations using execute_go
        # 2. Attempt to pick up an object using execute_pick
        # 3. Attempt to place the object using execute_place
        # 4. Attempt push/pull to interact with drawers
        # 5. Check for observed feedback

        # --- For demonstration, check for 'robot-free' feedback issue ---
        #
        # Root cause: something expects '(robot-free)' to be true before action and it isn't.
        # Let's explore via available skills, especially those that change 'robot-free'.
        #
        # Approach:
        #   - Try actions that produce/consume 'robot-free' (notably execute_pick, execute_place, execute_pull, execute_push)
        #   - If action fails, print status so user can analyze.
        #
        # We'll assume object and place keys correspond to the simulation.

        object_keys = [k for k in positions.keys() if 'object' in k or 'obj' in k]
        location_keys = [k for k in positions.keys() if 'loc' in k or 'location' in k or 'place' in k or 'room' in k]
        drawer_keys = [k for k in positions.keys() if 'drawer' in k]
        handle_keys = [k for k in positions.keys() if 'handle' in k]

        # Pick a sample object, location, drawer for demonstration
        sample_obj = object_keys[0] if object_keys else None
        sample_location = location_keys[0] if location_keys else None
        sample_drawer = drawer_keys[0] if drawer_keys else None
        sample_handle = handle_keys[0] if handle_keys else None

        # Step 1: Try moving to a location (execute_go)
        if len(location_keys) > 1:
            try:
                print(f"[Exploration] Trying to move from {location_keys[0]} to {location_keys[1]} (execute_go)")
                obs, reward, done = execute_go(
                    env=env,
                    task=task,
                    from_location=positions[location_keys[0]],
                    to_location=positions[location_keys[1]],
                    max_steps=80,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Moved successfully.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # Step 2: Try picking up a sample object (execute_pick)
        if sample_obj and sample_location:
            try:
                print(f"[Exploration] Trying to pick up {sample_obj} at {sample_location} (execute_pick)")
                obs, reward, done = execute_pick(
                    env=env,
                    task=task,
                    target=sample_obj,
                    place=positions[sample_location],
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] execute_pick success.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

        # Step 3: Try placing the object in a drawer (execute_place)
        if sample_obj and sample_drawer and sample_location:
            try:
                print(f"[Exploration] Trying to place {sample_obj} into {sample_drawer} at {sample_location} (execute_place)")
                obs, reward, done = execute_place(
                    env=env,
                    task=task,
                    target=sample_obj,
                    drawer=sample_drawer,
                    place=positions[sample_location],
                    approach_distance=0.12,
                    max_steps=100,
                    threshold=0.02,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] execute_place success.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # Step 4: Try pulling a drawer open (execute_pull)
        if sample_drawer and sample_handle and sample_location:
            try:
                print(f"[Exploration] Trying to pull open {sample_drawer} by {sample_handle} at {sample_location} (execute_pull)")
                obs, reward, done = execute_pull(
                    env=env,
                    task=task,
                    drawer=sample_drawer,
                    handle=sample_handle,
                    place=positions[sample_location],
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.02,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] execute_pull success.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")
        
        # Step 5: Try pushing a drawer closed (execute_push)
        if sample_drawer and sample_location:
            try:
                print(f"[Exploration] Trying to push closed {sample_drawer} at {sample_location} (execute_push)")
                obs, reward, done = execute_push(
                    env=env,
                    task=task,
                    drawer=sample_drawer,
                    place=positions[sample_location],
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.02,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] execute_push success.")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # If exploration finds 'robot-free' missing from state during any action (e.g., error message, failed precondition)
        # it will be printed to the user for diagnosis.
        print("[Exploration] Exploration phase completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
