# run_skeleton_task.py (Completed as per instructions)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only pre-defined primitive skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ===================================================================
        # EXPLORATION PHASE: Try combinations to discover possible missing predicates or facts
        # ===================================================================
        print("[Exploration] Beginning exploration to discover missing predicates...")

        # List all available skills for exploration attempts
        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Find out what objects, drawers, handles, locations are present (using descriptions or positions)
        # Assumes positions: {'objects': {...}, 'drawers': {...}, 'handles': {...}, 'locations': {...}}
        objects = positions.get('objects', {})
        drawers = positions.get('drawers', {})
        handles = positions.get('handles', {})
        locations = positions.get('locations', {})

        # Use a generic exploration strategy: for each available skill, try featurally plausible invocations
        # Record failures and findings for developer analysis

        exploration_results = []

        # For demonstration: try to pick each object at each location
        for obj_name, obj_pos in objects.items():
            for loc_name, loc_pos in locations.items():
                try:
                    print(f"[Exploration] Trying execute_pick on {obj_name} at {loc_name}")
                    obs, reward, done = execute_pick(env, task, target=obj_name, location=loc_name)
                    exploration_results.append(('execute_pick', obj_name, loc_name, reward, done))
                except Exception as e:
                    exploration_results.append(('execute_pick_fail', obj_name, loc_name, str(e)))
                    print(f"[Exploration] execute_pick failed on {obj_name} at {loc_name}: {e}")

        # Try pulling every handle at its associated drawer/location
        for h_name, h_pos in handles.items():
            # Attempt to match handle to a drawer if mapping exists
            matched_drawer_name = None
            for d_name in drawers:
                # Suppose handle and drawer mapping is inferred by naming convention or direct mapping
                if h_name in d_name or d_name in h_name:
                    matched_drawer_name = d_name
                    break
            if not matched_drawer_name:
                continue # Skip if no matching drawer found

            for loc_name in locations:
                try:
                    print(f"[Exploration] Trying execute_pull on {matched_drawer_name}, handle {h_name} at {loc_name}")
                    obs, reward, done = execute_pull(env, task, drawer=matched_drawer_name, handle=h_name, location=loc_name)
                    exploration_results.append(('execute_pull', matched_drawer_name, h_name, loc_name, reward, done))
                except Exception as e:
                    exploration_results.append(('execute_pull_fail', matched_drawer_name, h_name, loc_name, str(e)))
                    print(f"[Exploration] execute_pull failed on {matched_drawer_name}, handle {h_name} at {loc_name}: {e}")

        # Try go actions between locations (assuming at least two locations exist)
        loc_names = list(locations.keys())
        if len(loc_names) > 1:
            for i in range(len(loc_names)):
                for j in range(len(loc_names)):
                    if i == j:
                        continue
                    try:
                        print(f"[Exploration] Trying execute_go from {loc_names[i]} to {loc_names[j]}")
                        obs, reward, done = execute_go(env, task, from_location=loc_names[i], to_location=loc_names[j])
                        exploration_results.append(('execute_go', loc_names[i], loc_names[j], reward, done))
                    except Exception as e:
                        exploration_results.append(('execute_go_fail', loc_names[i], loc_names[j], str(e)))
                        print(f"[Exploration] execute_go failed from {loc_names[i]} to {loc_names[j]}: {e}")

        # Try to push (close) each drawer at each location
        for d_name in drawers:
            for loc_name in locations:
                try:
                    print(f"[Exploration] Trying execute_push on {d_name} at {loc_name}")
                    obs, reward, done = execute_push(env, task, drawer=d_name, location=loc_name)
                    exploration_results.append(('execute_push', d_name, loc_name, reward, done))
                except Exception as e:
                    exploration_results.append(('execute_push_fail', d_name, loc_name, str(e)))
                    print(f"[Exploration] execute_push failed on {d_name} at {loc_name}: {e}")

        # Try to place every object in every drawer at every location
        for obj_name in objects:
            for d_name in drawers:
                for loc_name in locations:
                    try:
                        print(f"[Exploration] Trying execute_place for {obj_name} in {d_name} at {loc_name}")
                        obs, reward, done = execute_place(env, task, target=obj_name, drawer=d_name, location=loc_name)
                        exploration_results.append(('execute_place', obj_name, d_name, loc_name, reward, done))
                    except Exception as e:
                        exploration_results.append(('execute_place_fail', obj_name, d_name, loc_name, str(e)))
                        print(f"[Exploration] execute_place failed for {obj_name} in {d_name} at {loc_name}: {e}")

        # Sweep: for each object at each location, attempt to sweep (usually a fallback or test)
        for obj_name in objects:
            for loc_name in locations:
                try:
                    print(f"[Exploration] Trying execute_sweep for {obj_name} at {loc_name}")
                    obs, reward, done = execute_sweep(env, task, target=obj_name, location=loc_name)
                    exploration_results.append(('execute_sweep', obj_name, loc_name, reward, done))
                except Exception as e:
                    exploration_results.append(('execute_sweep_fail', obj_name, loc_name, str(e)))
                    print(f"[Exploration] execute_sweep failed for {obj_name} at {loc_name}: {e}")

        # Try execute_gripper if applicable
        try:
            print(f"[Exploration] Trying execute_gripper (general skill)")
            obs, reward, done = execute_gripper(env, task)
            exploration_results.append(('execute_gripper', reward, done))
        except Exception as e:
            exploration_results.append(('execute_gripper_fail', str(e)))
            print(f"[Exploration] execute_gripper failed: {e}")

        # Optional: Display summary of results for developer
        print("[Exploration Finished] Exploration results:")
        for r in exploration_results:
            print(r)

        print("[Exploration] Completed. If fails or errors persist, check predicates in domain/problem PDDL based on failed skill calls.")
        # ========================
        # END EXPLORATION SECTION
        # ========================

        print("[Task] End of exploration phase. Implement oracle plan execution here if available.")
        # At this point, based on exploration and logs, a human can determine which predicate is missing or misconfigured.

        # Optionally: Further logic to attempt the oracle plan can be plugged here, reusing the available skills based on exploration insight.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
