# run_skeleton_task.py (Completed Executable Code for Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only pre-defined skills, do NOT redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # EXPLORATION PHASE: Try each available skill to probe what predicate may be missing
        # and gather information about predicates needed for the task.
        
        # Available skills (from provided list):
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        
        # We'll use object and location names from positions dict if possible.
        # Find one object, one location, one drawer, and one handle if available.
        obj_name = None
        loc_name = None
        drawer_name = None
        handle_name = None
        # Simple object/type heuristics
        for key in positions:
            if 'drawer' in key and drawer_name is None:
                drawer_name = key
            elif 'handle' in key and handle_name is None:
                handle_name = key
            elif obj_name is None and ('cube' in key or 'obj' in key):
                obj_name = key
            elif loc_name is None and ('loc' in key or 'table' in key or 'floor' in key):
                loc_name = key
        # In absence, default to some
        pos_keys = list(positions.keys())
        if obj_name is None and len(pos_keys) > 0:
            obj_name = pos_keys[0]
        if loc_name is None and len(pos_keys) > 1:
            loc_name = pos_keys[1]
        if drawer_name is None and len(pos_keys) > 2:
            drawer_name = pos_keys[2]
        if handle_name is None and len(pos_keys) > 3:
            handle_name = pos_keys[3] if len(pos_keys) > 3 else obj_name
        
        print("[Exploration] Using obj_name:", obj_name, "loc_name:", loc_name, "drawer_name:", drawer_name, "handle_name:", handle_name)
        observation_records = []
        
        # 1. Try execute_pick on floor object (simulate picking up an object)
        try:
            print("[Exploration] Attempting execute_pick...")
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name,
                loc_name
            )
            observation_records.append(('execute_pick', obs, reward, done))
            print("[Exploration] execute_pick executed.")
        except Exception as e:
            print("[Exploration] execute_pick failed:", e)
        
        # 2. Try execute_place into a drawer (simulate placing object in a drawer)
        try:
            print("[Exploration] Attempting execute_place...")
            obs, reward, done = execute_place(
                env,
                task,
                obj_name,
                drawer_name,
                loc_name
            )
            observation_records.append(('execute_place', obs, reward, done))
            print("[Exploration] execute_place executed.")
        except Exception as e:
            print("[Exploration] execute_place failed:", e)

        # 3. Try execute_push (simulate pushing to close drawer)
        try:
            print("[Exploration] Attempting execute_push...")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name,
                loc_name
            )
            observation_records.append(('execute_push', obs, reward, done))
            print("[Exploration] execute_push executed.")
        except Exception as e:
            print("[Exploration] execute_push failed:", e)

        # 4. Try execute_pull (simulate pulling to open a drawer)
        try:
            print("[Exploration] Attempting execute_pull...")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                loc_name
            )
            observation_records.append(('execute_pull', obs, reward, done))
            print("[Exploration] execute_pull executed.")
        except Exception as e:
            print("[Exploration] execute_pull failed:", e)

        # 5. Try execute_sweep (simulate sweep action)
        try:
            print("[Exploration] Attempting execute_sweep...")
            obs, reward, done = execute_sweep(
                env,
                task,
                obj_name,
                loc_name
            )
            observation_records.append(('execute_sweep', obs, reward, done))
            print("[Exploration] execute_sweep executed.")
        except Exception as e:
            print("[Exploration] execute_sweep failed:", e)
        
        # 6. Try execute_go (move robot from one location to another)
        # We need two locations; reusing loc_name and, if present, another
        second_loc = None
        for k in pos_keys:
            if k != loc_name:
                second_loc = k
                break
        if second_loc is None:
            second_loc = loc_name
        try:
            print("[Exploration] Attempting execute_go...")
            obs, reward, done = execute_go(
                env,
                task,
                loc_name,
                second_loc
            )
            observation_records.append(('execute_go', obs, reward, done))
            print("[Exploration] execute_go executed.")
        except Exception as e:
            print("[Exploration] execute_go failed:", e)
        
        # 7. Try execute_gripper (open/close the gripper)
        try:
            print("[Exploration] Attempting execute_gripper...")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            observation_records.append(('execute_gripper', obs, reward, done))
            print("[Exploration] execute_gripper executed.")
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        # 8. Try execute_rotate if available (not described in PDDL, but included in skills)
        if 'execute_rotate' in globals():
            try:
                print("[Exploration] Attempting execute_rotate...")
                obs, reward, done = execute_rotate(
                    env,
                    task,
                    obj_name,
                    loc_name
                )
                observation_records.append(('execute_rotate', obs, reward, done))
                print("[Exploration] execute_rotate executed.")
            except Exception as e:
                print("[Exploration] execute_rotate failed:", e)

        # The above explorations should help uncover if any predicates are missing in the initial state
        # and if the available skills correspond to the expected PDDL predicates and effects.

        print("\n[Exploration] Skill execution complete. Observed results:")
        for rec in observation_records:
            print("   Skill:", rec[0], "Reward:", rec[2], "Done:", rec[3])

        # === END of Exploration - place any further plan/goal-specific logic here ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
