# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All primitive skills are called directly as per the guidelines

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ---- Exploration Phase: Identify Missing Predicate ----
        # Based on feedback, the missing predicate is (robot-free)
        # We need to explicitly check the robot's free state in skill execution.

        # Example of exploration to verify the missing predicate logic
        print("[Exploration] Checking robot-free status...")
        try:
            # The actual 'robot-free' predicate is not provided as an observable in the code,
            # but we can simulate skill execution that requires robot-free as a precondition.
            # For instance, attempt to call a skill like execute_go that requires robot-free.

            # Let's assume we want to move the robot from its current position to some target location,
            # and per the PDDL, execute_go requires (robot-free) as a precondition.

            # Find robot and location from positions dictionary
            robot_pos = None
            loc_names = []
            for k, v in positions.items():
                # Example: 'robot' : (x, y, z)
                if 'robot' in k:
                    robot_pos = k
                elif 'location' in k or 'loc' in k or 'room' in k:
                    loc_names.append(k)
            # Fallback for example purposes
            position_keys = list(positions.keys())
            if len(position_keys) >= 2:
                from_location = position_keys[0]
                to_location = position_keys[1]
            else:
                # Not enough locations defined, skip movement exploration
                from_location = None
                to_location = None

            # Simulate: execute_go(from_location, to_location)
            if from_location and to_location and from_location != to_location:
                print("[Exploration] Trying to move robot (execute_go) from", from_location, "to", to_location)
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=positions[from_location],
                        to_location=positions[to_location],
                        max_steps=80,
                        approach_distance=0.15
                    )
                    print("[Exploration] execute_go succeeded.")
                except Exception as move_exc:
                    print("[Exploration] execute_go failed, possible missing robot-free precondition:", move_exc)
            else:
                print("[Exploration] Not enough locations available for move exploration.")

            # Simulate other skills that depend on robot-free if needed
            # (e.g., execute_pick, execute_push, etc.)
            # For illustration, attempt to execute a pick action if possible
            # Find an on-floor object and a location
            pick_object = None
            for name in positions:
                if 'object' in name or 'obj' in name:
                    pick_object = name
                    break
            if pick_object and from_location:
                print("[Exploration] Trying to pick object (execute_pick):", pick_object, "at", from_location)
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=pick_object,
                        location=positions[from_location],
                        approach_distance=0.10,
                        max_steps=60
                    )
                    print("[Exploration] execute_pick succeeded.")
                except Exception as pick_exc:
                    print("[Exploration] execute_pick failed, possibly due to robot-free or hand-empty state:", pick_exc)
            else:
                print("[Exploration] Not enough objects/locations for pick exploration.")

        except Exception as exploration_exc:
            print("[Exploration] General exploration error:", exploration_exc)

        # ---- End of Exploration Phase ----

        # === Main Plan Execution (Oracle Plan if provided) ===
        # ---- Example: Manual plan execution step-by-step ----
        # For this template, suppose the overall goal is to pick up an object and place it into a drawer.
        # We'll outline the plan steps and call appropriate skills.

        # 1. Move robot to object location (if not already there)
        # 2. Pick up the object
        # 3. Move robot to the drawer location
        # 4. If drawer is closed/unlocked and handle known, pull handle to open
        # 5. Place object into the drawer

        # === (1) Move to object location ===
        # Suppose we have an object and a location; get these from positions
        main_object = None
        object_loc = None
        for name, pos in positions.items():
            if 'object' in name or 'obj' in name:
                main_object = name
                # Let's suppose object's location key is also available
                object_loc = name + '_loc' if name + '_loc' in positions else None
                break

        if not main_object:
            print("[Task] No main object found to manipulate.")
        else:
            # If location is missing, just use any 'room' or location name from positions
            if not object_loc:
                for k in positions:
                    if 'room' in k or 'loc' in k:
                        object_loc = k
                        break

            # Move to object location
            if object_loc:
                try:
                    print("[Task] Moving to object location:", object_loc)
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=None,       # Assume the skill uses current location
                        to_location=positions[object_loc],
                        max_steps=80,
                        approach_distance=0.15
                    )
                except Exception as go_exc:
                    print("[Task] Move to object location failed:", go_exc)

                # === (2) Pick up the object ===
                try:
                    print("[Task] Picking up object:", main_object)
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=main_object,
                        location=positions[object_loc],
                        approach_distance=0.10,
                        max_steps=60
                    )
                    if done:
                        print("[Task] Pick action completed.")
                except Exception as pick_exc:
                    print("[Task] Pick failed:", pick_exc)
            else:
                print("[Task] No object location identified. Skipping pick.")

        # === (3) Move to drawer location ===
        # Try to find a drawer and location
        drawer_name = None
        drawer_loc = None
        for name in positions:
            if 'drawer' in name:
                drawer_name = name
                # Drawer location key
                drawer_loc = name + '_loc' if name + '_loc' in positions else None
                break
        if not drawer_loc:
            for k in positions:
                if 'room' in k or 'loc' in k:
                    drawer_loc = k
                    break

        if drawer_loc:
            try:
                print("[Task] Moving to drawer location:", drawer_loc)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=None,
                    to_location=positions[drawer_loc],
                    max_steps=80,
                    approach_distance=0.15
                )
            except Exception as go2_exc:
                print("[Task] Move to drawer location failed:", go2_exc)
        else:
            print("[Task] No drawer location recognized. Skipping drawer approach.")

        # === (4) Open drawer if needed (if skill and handles are available) ===
        # The actual handle logic may rely on specific predicates, but for now we attempt if the skill exists
        try:
            if drawer_name and main_object and drawer_loc:
                print("[Task] Trying to open the drawer using:", main_object)
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_object=main_object,
                    location=positions[drawer_loc],
                    approach_distance=0.10,
                    max_steps=60
                )
                print("[Task] Drawer pulled (opened).")
        except Exception as pull_exc:
            print("[Task] Drawer opening failed (might be already open, wrong handle, or locked):", pull_exc)

        # === (5) Place the object into the drawer ===
        try:
            if main_object and drawer_name and drawer_loc:
                print("[Task] Placing object into drawer...")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=main_object,
                    drawer_name=drawer_name,
                    location=positions[drawer_loc],
                    approach_distance=0.10,
                    max_steps=60
                )
                print("[Task] Place into drawer succeeded.")
        except Exception as place_exc:
            print("[Task] Place into drawer failed:", place_exc)

        # Optionally, push (close) the drawer
        try:
            if drawer_name and drawer_loc:
                print("[Task] Closing the drawer...")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer_name,
                    location=positions[drawer_loc],
                    approach_distance=0.10,
                    max_steps=60
                )
                print("[Task] Drawer closed.")
        except Exception as push_exc:
            print("[Task] Drawer close failed:", push_exc)

        # Task completion printout
        print("[Task] Task logic completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
